package ru.yandex.partner.jsonapi.models.block.rtb.mobile.internal;

import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import ru.yandex.partner.core.entity.block.model.InternalMobileRtbBlock;
import ru.yandex.partner.core.entity.dsp.service.DspService;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.InternalMobileApp;
import ru.yandex.partner.core.entity.page.service.ReachablePageService;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.QueryParamsContext;
import ru.yandex.partner.jsonapi.models.block.BlockRightNames;
import ru.yandex.partner.jsonapi.models.block.access.NotCustomBkDataAccessRuleFunction;
import ru.yandex.partner.jsonapi.models.block.parts.ApiDspsModelPart;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;


public class ApiInternalMobileRtbDspsModelPart extends ApiDspsModelPart<InternalMobileRtbBlock> {

    private final DspService dspService;
    private final ReachablePageService<InternalMobileRtbBlock> reachablePageService;

    public ApiInternalMobileRtbDspsModelPart(String modelName, DspService dspService,
                                             ReachablePageService<InternalMobileRtbBlock> reachablePageService) {
        super(modelName);
        this.dspService = dspService;
        this.reachablePageService = reachablePageService;
    }

    @Override
    protected ApiFieldsAccessRulesFunction<InternalMobileRtbBlock> isUnmoderatedRtbAuctionFunction() {
        return new ApiFieldsAccessRulesFunction<>((userAuthentication, rtbBlock) ->
                Boolean.TRUE.equals(rtbBlock.getCampaign().getUnmoderatedRtbAuction()),
                InternalMobileRtbBlock.CAMPAIGN
        );

    }

    @Override
    protected ApiFieldsAccessRulesFunction<InternalMobileRtbBlock> isBlockDspsUnmoderatedAvailableFunction() {
        return new ApiFieldsAccessRulesFunction<>(
                (userAuthentication, rtbBlock) ->
                        userAuthentication.userHasRight(
                                BlockRightNames.VIEW_FIELD__UNMODERATED_DSPS.getFullRightName(getModelName()))
                                && rtbBlock.getCampaign() != null
                                && Boolean.TRUE.equals(rtbBlock.getCampaign().getUnmoderatedRtbAuction()),
                InternalMobileRtbBlock.CAMPAIGN);
    }

    @Override
    public ApiFieldsAccessRulesFunction<InternalMobileRtbBlock> isGodModeFunction() {
        return new NotCustomBkDataAccessRuleFunction<>();
    }

    @Override
    protected Function<QueryParamsContext<InternalMobileRtbBlock>, Map<String, Object>> getDspsDefaultsFunction() {
        return context -> {
            InternalMobileRtbBlock block = context.getModelFromAttributes();
            InternalMobileApp campaign = Optional.ofNullable(block.getPageId())
                    .map(pageId -> CoreFilterNode.eq(PageFilters.PAGE_ID, pageId))
                    .map(filter -> reachablePageService.getReachablePages(filter, InternalMobileApp.class))
                    .orElseThrow(() -> new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS))
                    .stream().findFirst()
                    .orElseThrow(() -> new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_FOUND));
            block.setCampaign(campaign);

            UserAuthentication user = context.getUserAuthentication();
            boolean userHasRight = user.userHasRight(BlockRightNames.VIEW_FIELD__DSPS.getFullRightName(getModelName()));
            boolean canEditRichMedia = user.userHasRight(BlockRightNames.EDIT_FIELD__RICH_MEDIA
                    .getFullRightName(getModelName()));
            return dspService.getDefaults(block, userHasRight, campaign.getUnmoderatedRtbAuction(), canEditRichMedia);
        };
    }
}
