package ru.yandex.partner.jsonapi.models.designtemplates;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import io.crnk.core.engine.information.resource.ResourceFieldType;

import ru.yandex.partner.core.entity.designtemplates.filter.DesignTemplatesFilters;
import ru.yandex.partner.core.entity.designtemplates.model.CommonDesignTemplates;
import ru.yandex.partner.dbschema.partner.enums.DesignTemplatesType;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilter;
import ru.yandex.partner.jsonapi.models.ApiModelPart;
import ru.yandex.partner.jsonapi.models.ModelClassUtils;
import ru.yandex.partner.jsonapi.utils.function.BatchBiFunction;
import ru.yandex.partner.jsonapi.utils.function.BatchFunction;

@ParametersAreNonnullByDefault
public class ApiDesignTemplatesModelPart<D extends CommonDesignTemplates> implements ApiModelPart<D> {
    private static final String ID = "id";
    private static final String BLOCK_ID = "block_id";
    private static final String PAGE_ID = "page_id";
    private static final String TYPE = "type";
    private static final String CAPTION = "caption";
    private static final String DESIGN_SETTINGS = "design_settings";
    private static final String FILTER_TAGS = "filter_tags";
    private static final String CUSTOM_FORMAT_DIRECT = "custom_format_direct";
    private static final String PCODE_KEY = "pcode_key";
    private static final String NATIVE_TEMPLATE_FACTORS = "native_template_factors";


    @Override
    public List<ApiField<D>> getFields() {
        return List.of(
                ApiField
                        .<D, Long>convertibleApiField(Long.class, D.ID)
                        .withResourceFieldType(ResourceFieldType.ID)
                        .build(ID),
                ApiField.<D, Long>convertibleApiField(Long.class, D.BLOCK_ID)
                        .build(BLOCK_ID),
                ApiField.<D, Long>convertibleApiField(Long.class, D.PAGE_ID)
                        .build(PAGE_ID),
                ApiField.<D, String, DesignTemplatesType>convertibleApiField(String.class, D.TYPE,
                                BatchFunction.one(it -> it.getType().getLiteral()),
                                BatchBiFunction.one((designTemplates, type) -> {
                                    switch (type) {
                                        case "media":
                                            designTemplates.setType(DesignTemplatesType.media);
                                            break;
                                        case "native":
                                            designTemplates.setType(DesignTemplatesType.native_);
                                            break;
                                        case "tga":
                                            designTemplates.setType(DesignTemplatesType.tga);
                                            break;
                                        case "video":
                                            designTemplates.setType(DesignTemplatesType.video);
                                            break;
                                        default:
                                            throw new IllegalArgumentException("wrong type");
                                    }
                                    return null;
                                }))
                        .build(TYPE),
                ApiField.<D, String>convertibleApiField(String.class, D.CAPTION)
                        .build(CAPTION),
                ApiField.<D, List<String>>convertibleApiField(ModelClassUtils.getListClass(String.class), D.FILTER_TAGS)
                        .build(FILTER_TAGS),
                ApiField.<D, Boolean>convertibleApiField(Boolean.class, D.CUSTOM_FORMAT_DIRECT)
                        .build(CUSTOM_FORMAT_DIRECT),
                ApiField.<D, Map<String, Object>>convertibleApiField(ModelClassUtils.getMapClass(String.class,
                                Object.class), D.DESIGN_SETTINGS)
                        .build(DESIGN_SETTINGS),
                ApiField.<D, Map<String, Object>>convertibleApiField(ModelClassUtils.getMapClass(String.class,
                                Object.class), D.NATIVE_TEMPLATE_FACTORS)
                        .build(NATIVE_TEMPLATE_FACTORS),
                ApiField.<D, String>convertibleApiField(String.class, D.PCODE_KEY)
                        .build(PCODE_KEY)
        );
    }

    @Override
    public Map<String, CrnkFilter<D, ?>> getFilters() {
        List<CrnkFilter<D, ?>> filtersList = List.of(
                CrnkFilter.builder(DesignTemplatesFilterNameEnum.BLOCK_ID)
                        .toCrnkFilter(DesignTemplatesFilters.BLOCK_ID),
                CrnkFilter.builder(DesignTemplatesFilterNameEnum.PAGE_ID)
                        .toCrnkFilter(DesignTemplatesFilters.PAGE_ID),
                CrnkFilter.builder(DesignTemplatesFilterNameEnum.ID)
                        .toCrnkFilter(DesignTemplatesFilters.ID),
                CrnkFilter.builder(DesignTemplatesFilterNameEnum.MULTISTATE)
                        .toCrnkFilter(DesignTemplatesFilters.MULTISTATE),
                CrnkFilter.builder(DesignTemplatesFilterNameEnum.CAPTION)
                        .toCrnkFilter(DesignTemplatesFilters.CAPTION)
        );

        return filtersList.stream()
                .collect(Collectors.toMap(
                        CrnkFilter::getName, it -> it
                ));
    }
}
