package ru.yandex.partner.jsonapi.models.user.parts;

import java.util.List;
import java.util.Objects;
import java.util.function.BiConsumer;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.EditableData;
import ru.yandex.partner.jsonapi.crnk.user.UserRightsEnum;

import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.checkable;

@ParametersAreNonnullByDefault
public class ApiModelAvailability {

    private static final ApiFieldsAccessRulesFunction<User> SELF_VIEW =
            new ApiFieldsAccessRulesFunction<>(
                    (userAuthentication, user) -> Objects.equals(userAuthentication.getUid(), user.getId()));

    private static final ApiFieldsAccessRulesFunction<User> CAN_VIEW_ALL_FIELDS =
            new ApiFieldsAccessRulesFunction<>(
                    (userAuthentication, user) -> userAuthentication.userHasRight(UserRightsEnum.VIEW_ALL_FIELDS));

    private static final ApiFieldsAccessRulesFunction<User> SELF_VIEW_OR_CAN_VIEW_ALL_FIELDS =
            ApiFieldsAccessRulesFunction.or(List.of(SELF_VIEW, CAN_VIEW_ALL_FIELDS));

    private static final ApiFieldsAccessRulesFunction<EditableData<User>> SELF_EDIT_OR_CAN_EDIT_ALL_FIELDS =
            new ApiFieldsAccessRulesFunction<>(
                    (userAuthentication1, user1) ->
                            Objects.equals(userAuthentication1.getUid(), user1.getActual().getId())
                                    || userAuthentication1.userHasRight(UserRightsEnum.EDIT_ALL),
                    User.ID);


    private ApiModelAvailability() {
        // utility
    }

    public static ApiFieldsAccessRulesFunction<User>
    selfViewOrCanViewAllFieldsAndHasRight(UserRightsEnum userRightsEnum) {
        return ApiFieldsAccessRulesFunction.and(List.of(
                selfViewOrCanViewAllFields(),
                new ApiFieldsAccessRulesFunction<>(
                        (userAuthentication, model) -> userAuthentication.userHasRight(userRightsEnum))));
    }

    public static BiConsumer<String, ApiFieldsAccessRules.Builder<User>>
    selfViewOrCanViewAllFieldsAndHasRightConsumer(UserRightsEnum userRightsEnum) {

        return checkable(selfViewOrCanViewAllFieldsAndHasRight(userRightsEnum));

    }

    public static ApiFieldsAccessRulesFunction<User> selfView() {
        return SELF_VIEW;
    }

    public static ApiFieldsAccessRulesFunction<User> canViewAllFields() {
        return CAN_VIEW_ALL_FIELDS;
    }

    public static ApiFieldsAccessRulesFunction<User> selfViewOrCanViewAllFields() {
        return SELF_VIEW_OR_CAN_VIEW_ALL_FIELDS;
    }

    public static ApiFieldsAccessRulesFunction<EditableData<User>> selfEditOrCanEditAllFields() {
        return SELF_EDIT_OR_CAN_EDIT_ALL_FIELDS;
    }

}
