package ru.yandex.partner.jsonapi.models.user.parts;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import io.crnk.core.engine.information.resource.ResourceFieldType;
import org.springframework.context.MessageSource;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.MessageSourceAccessor;

import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.dictionary.moderationreason.ModerationReasonService;
import ru.yandex.partner.core.entity.user.filter.UserFilters;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.entity.user.multistate.UserMultistateExpressionParser;
import ru.yandex.partner.core.entity.user.multistate.UserMultistateGraph;
import ru.yandex.partner.core.feature.UserFeatureEnum;
import ru.yandex.partner.core.service.integration.balance.BalanceDocumentService;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.EditableData;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilter;
import ru.yandex.partner.jsonapi.crnk.filter.description.provider.ModerationReasonDictionaryValuesProvider;
import ru.yandex.partner.jsonapi.crnk.filter.description.provider.RolesDictionaryValuesProvider;
import ru.yandex.partner.jsonapi.crnk.filter.description.provider.UserTypeDictionaryValuesProvider;
import ru.yandex.partner.jsonapi.crnk.filter.expose.CheckRightsCrnkFilterExposeStrategy;
import ru.yandex.partner.jsonapi.crnk.filter.parser.values.ContractNumberCrnkFilterValueParser;
import ru.yandex.partner.jsonapi.crnk.filter.parser.values.MultistateCrnkFilterValueParser;
import ru.yandex.partner.jsonapi.crnk.user.ActiveContractCrnkMapper;
import ru.yandex.partner.jsonapi.crnk.user.AdfoxInfoCrnkMapper;
import ru.yandex.partner.jsonapi.crnk.user.RoleCrnkMapper;
import ru.yandex.partner.jsonapi.crnk.user.UserCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.user.UserRightsEnum;
import ru.yandex.partner.jsonapi.crnk.user.filter.parser.UserTypeCrnkFilterValueParser;
import ru.yandex.partner.jsonapi.crnk.user.models.ActiveContractCrnk;
import ru.yandex.partner.jsonapi.crnk.user.models.AdfoxInfoCrnk;
import ru.yandex.partner.jsonapi.crnk.user.models.RoleCrnk;
import ru.yandex.partner.jsonapi.messages.ModerationReasonMsg;
import ru.yandex.partner.jsonapi.messages.PersonMsg;
import ru.yandex.partner.jsonapi.messages.user.UserMsg;
import ru.yandex.partner.jsonapi.models.ApiModelPart;
import ru.yandex.partner.jsonapi.models.ModelClassUtils;
import ru.yandex.partner.jsonapi.models.block.access.HasRightAccessRuleFunction;
import ru.yandex.partner.jsonapi.utils.function.BatchBiFunction;
import ru.yandex.partner.jsonapi.utils.function.BatchFunction;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.rbac.role.RoleSet;

import static ru.yandex.partner.core.entity.user.model.prop.UserWithRolesRolesPropHolder.ROLES;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.checkable;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.checkableByRight;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.sameAs;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction.checkableAnd;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction.checkableOr;
import static ru.yandex.partner.jsonapi.crnk.user.UserRightsEnum.EDIT_FIELD__IS_EFIR_BLOGGER;
import static ru.yandex.partner.jsonapi.crnk.user.UserRightsEnum.MODERATION_REASON_MANAGER_TXT_VIEW;
import static ru.yandex.partner.jsonapi.models.user.parts.ApiModelAvailability.canViewAllFields;
import static ru.yandex.partner.jsonapi.models.user.parts.ApiModelAvailability.selfEditOrCanEditAllFields;
import static ru.yandex.partner.jsonapi.models.user.parts.ApiModelAvailability.selfView;
import static ru.yandex.partner.jsonapi.models.user.parts.ApiModelAvailability.selfViewOrCanViewAllFields;
import static ru.yandex.partner.jsonapi.models.user.parts.ApiModelAvailability.selfViewOrCanViewAllFieldsAndHasRight;
import static ru.yandex.partner.jsonapi.models.user.parts.ApiModelAvailability.selfViewOrCanViewAllFieldsAndHasRightConsumer;

@ParametersAreNonnullByDefault
public class ApiUserPart implements ApiModelPart<User> {

    private static final Pattern UR_TYPE_PATTERN = Pattern.compile("^(?:byu|sw_yt|sw_ur|ua|yt|ur)$");
    private static final Pattern PH_TYPE_PATTERN = Pattern.compile("^(?:ph|sw_ytph|ytph)$");
    private final ModerationReasonService moderationReasonService;
    private final AuthenticationFacade authenticationFacade;
    private final UserMultistateGraph userMultistateGraph;
    private final ActiveContractCrnkMapper activeContractCrnkMapper;
    private final AdfoxInfoCrnkMapper adfoxInfoCrnkMapper;
    private final RoleCrnkMapper roleCrnkMapper;
    private final UserMultistateExpressionParser userMultistateExpressionParser;
    private final RolesDictionaryValuesProvider rolesDictionaryValuesProvider;
    private final BalanceDocumentService balanceDocumentService;
    private final UserTypeDictionaryValuesProvider userTypeDictionaryValuesProvider;
    private final ModerationReasonDictionaryValuesProvider moderationReasonDictionaryValuesProvider;
    private final MessageSourceAccessor messageSource;

    @SuppressWarnings("checkstyle:parameternumber")
    public ApiUserPart(ModerationReasonService moderationReasonService,
                       AuthenticationFacade authenticationFacade,
                       UserMultistateGraph userMultistateGraph,
                       ActiveContractCrnkMapper activeContractCrnkMapper,
                       AdfoxInfoCrnkMapper adfoxInfoCrnkMapper, RoleCrnkMapper roleCrnkMapper,
                       UserMultistateExpressionParser userMultistateExpressionParser,
                       RolesDictionaryValuesProvider rolesDictionaryValuesProvider,
                       BalanceDocumentService balanceDocumentService,
                       UserTypeDictionaryValuesProvider userTypeDictionaryValuesProvider,
                       ModerationReasonDictionaryValuesProvider moderationReasonDictionaryValuesProvider,
                       MessageSource messageSource) {
        this.moderationReasonService = moderationReasonService;
        this.authenticationFacade = authenticationFacade;
        this.userMultistateGraph = userMultistateGraph;
        this.activeContractCrnkMapper = activeContractCrnkMapper;
        this.adfoxInfoCrnkMapper = adfoxInfoCrnkMapper;
        this.roleCrnkMapper = roleCrnkMapper;
        this.userMultistateExpressionParser = userMultistateExpressionParser;
        this.rolesDictionaryValuesProvider = rolesDictionaryValuesProvider;
        this.balanceDocumentService = balanceDocumentService;
        this.userTypeDictionaryValuesProvider = userTypeDictionaryValuesProvider;
        this.moderationReasonDictionaryValuesProvider = moderationReasonDictionaryValuesProvider;
        this.messageSource = new MessageSourceAccessor(messageSource);
    }

    @Override
    public List<ApiField<User>> getFields() {
        var apiFieldExcludedDomains = getExcludedDomains();
        var apiFieldName = getName();
        var listOfParts = List.of(
                getMainFields(apiFieldName, apiFieldExcludedDomains),
                getPayoneerFields(apiFieldName),
                getCurrencyFields()
        );
        return listOfParts.stream()
                .flatMap(List::stream)
                .collect(Collectors.toList());
    }

    @SuppressWarnings("MethodLength")
    private List<ApiField<User>> getMainFields(ApiField<User> apiFieldName, ApiField<User> apiFieldExcludedDomains) {
        var idField = ApiField.<User, Long>convertibleApiField(Long.class, User.ID)
                .withResourceFieldType(ResourceFieldType.ID).withAvailableFunction(
                        selfViewOrCanViewAllFieldsAndHasRightConsumer(UserRightsEnum.VIEW_FIELD__USER_ID))
                .build(UserCrnkJsonFieldAliases.ID);

        return List.of(
                ApiField.<User, String>convertibleApiField(String.class, User.AVATAR)
                        .build(UserCrnkJsonFieldAliases.AVATAR),
                ApiField.<User, String>convertibleApiField(String.class, User.LOGIN)
                        .build(UserCrnkJsonFieldAliases.LOGIN),
                ApiField.<User, List<String>>convertibleApiField(ModelClassUtils.getListClass(String.class),
                                User.FEATURES)
                        .withEditableFunction(checkableByRight(UserRightsEnum.EDIT_FIELD__FEATURES))
                        .build(UserCrnkJsonFieldAliases.FEATURES),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.ALLOWED_DESIGN_AUCTION_NATIVE_ONLY)
                        .withEditableFunction(checkableByRight(
                                UserRightsEnum.EDIT_FIELD__ALLOWED_DESIGN_AUCTION_NATIVE_ONLY))
                        .build(UserCrnkJsonFieldAliases.ALLOWED_DESIGN_AUCTION_NATIVE_ONLY),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.CONTENT_BLOCK_EDIT_TEMPLATE_ALLOWED)
                        .withEditableFunction(
                                checkableByRight(UserRightsEnum.EDIT_FIELD__CONTENT_BLOCK_EDIT_TEMPLATE_ALLOWED))
                        .build(UserCrnkJsonFieldAliases.CONTENT_BLOCK_EDIT_TEMPLATE_ALLOWED),
                idField,
                ApiField.<User, Long>convertibleApiField(Long.class, User.UID)
                        .withAvailableFunction(sameAs(idField))
                        .build(UserCrnkJsonFieldAliases.UID),
                ApiField.<User, Long>convertibleApiField(Long.class, User.CLIENT_ID)
                        .withAvailableFunction(
                                selfViewOrCanViewAllFieldsAndHasRightConsumer(UserRightsEnum.VIEW_FIELD__CLENT_ID))
                        .withEditableFunction(checkableByRight(UserRightsEnum.EDIT_FIELD__CLIENT_ID))
                        .build(UserCrnkJsonFieldAliases.CLIENT_ID),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.IS_DM_LITE)
                        .withAvailableFunction(
                                selfViewOrCanViewAllFieldsAndHasRightConsumer(UserRightsEnum.VIEW_FIELD__IS_DM_LITE))
                        .withEditableFunction(checkableByRight(UserRightsEnum.EDIT_FIELD__IS_DM_LITE))
                        .build(UserCrnkJsonFieldAliases.IS_DM_LITE),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.IS_EFIR_BLOGGER)
                        .withAvailableFunction(selfViewOrCanViewAllFieldsAndHasRightConsumer(
                                UserRightsEnum.VIEW_FIELD__IS_EFIR_BLOGGER))
                        .withEditableFunction(checkableByRight(EDIT_FIELD__IS_EFIR_BLOGGER))
                        .build(UserCrnkJsonFieldAliases.IS_EFIR_BLOGGER),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.IS_GAMES)
                        .withAvailableFunction(
                                selfViewOrCanViewAllFieldsAndHasRightConsumer(UserRightsEnum.VIEW_FIELD__IS_GAMES))
                        .withEditableFunction(checkableByRight(UserRightsEnum.EDIT_FIELD__IS_GAMES))
                        .build(UserCrnkJsonFieldAliases.IS_GAMES),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.IS_MOBILE_MEDIATION)
                        .withAvailableFunction(selfViewOrCanViewAllFieldsAndHasRightConsumer(
                                UserRightsEnum.VIEW_FIELD__IS_MOBILE_MEDIATION))
                        .withEditableFunction(checkableByRight(UserRightsEnum.EDIT_FIELD__IS_MOBILE_MEDIATION))
                        .build(UserCrnkJsonFieldAliases.IS_MOBILE_MEDIATION),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.IS_TUTBY)
                        .withAvailableFunction(
                                selfViewOrCanViewAllFieldsAndHasRightConsumer(UserRightsEnum.VIEW_FIELD__IS_TUTBY))
                        .withEditableFunction(checkableByRight(UserRightsEnum.EDIT_FIELD__IS_TUTBY))
                        .build(UserCrnkJsonFieldAliases.IS_TUTBY),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.IS_VIDEO_BLOGGER)
                        .withAvailableFunction(selfViewOrCanViewAllFieldsAndHasRightConsumer(
                                UserRightsEnum.VIEW_FIELD__IS_VIDEO_BLOGGER))
                        .withEditableFunction(checkableByRight(UserRightsEnum.EDIT_FIELD__IS_VIDEO_BLOGGER))
                        .build(UserCrnkJsonFieldAliases.IS_VIDEO_BLOGGER),
                getRolesApiField(),
                apiFieldExcludedDomains,
                ApiField.<User, List<String>>convertibleApiField(ModelClassUtils.getListClass(String.class),
                                User.EXCLUDED_PHONES)
                        .withAvailableFunction(sameAs(apiFieldExcludedDomains))
                        .withEditableFunction(sameAs(apiFieldExcludedDomains))
                        .build(UserCrnkJsonFieldAliases.EXCLUDED_PHONES),
                apiFieldName,
                ApiField.<User, String>convertibleApiField(String.class, User.EMAIL)
                        .withAvailableFunction(ApiFieldsAccessRulesFunction.checkableOr(List.of(
                                selfView(),
                                ApiFieldsAccessRulesFunction.and(List.of(
                                        canViewAllFields(),
                                        new HasRightAccessRuleFunction<>(
                                                UserRightsEnum.VIEW_FIELD__EMAIL.getRightName())
                                ))
                        )))
                        .withEditableFunction(sameAs(apiFieldName))
                        .build(UserCrnkJsonFieldAliases.EMAIL),
                ApiField.<User, String>convertibleApiField(String.class, User.LASTNAME)
                        .withAvailableFunction(sameAs(apiFieldName))
                        .withEditableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.LAST_NAME),
                ApiField.<User, String>convertibleApiField(String.class, User.MIDNAME)
                        .withAvailableFunction(sameAs(apiFieldName))
                        .withEditableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.MID_NAME),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.IS_ASSESSOR)
                        .withAvailableFunction(sameAs(apiFieldName))
                        .build(UserCrnkJsonFieldAliases.IS_ASSESSOR),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.NEWSLETTER)
                        .withAvailableFunction(sameAs(apiFieldName))
                        .withEditableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.NEWSLETTER),
                ApiField.<User, String>convertibleApiField(String.class, User.PHONE)
                        .withAvailableFunction(sameAs(apiFieldName))
                        .withEditableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.PHONE),
                ApiField.<User, String>convertibleApiField(String.class, User.STATUS)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.STATUS),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.NO_STAT_MONITORING_EMAILS)
                        .withAvailableFunction(sameAs(apiFieldName))
                        .withEditableFunction(sameAs(apiFieldName))
                        .build(UserCrnkJsonFieldAliases.NO_STAT_MONITORING_EMAILS),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.BLOCK_LIGHT_FORM_ENABLED)
                        .withAvailableFunction(sameAs(apiFieldName))
                        .withEditableFunction(sameAs(apiFieldName))
                        .build(UserCrnkJsonFieldAliases.BLOCK_LIGHT_FORM_ENABLED),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.IS_ADFOX_PARTNER)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.IS_ADFOX_PARTNER),
                ApiField.<User, String>convertibleApiField(String.class, User.FULL_NAME)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.FULL_NAME),
                ApiField.<User, LocalDateTime, LocalDateTime>convertibleApiField(
                                LocalDateTime.class, User.LAST_PAYOUT,
                                BatchFunction.one(user -> user.getLastPayout() != null ? user.getLastPayout() :
                                        user.getCreateDate()),
                                BatchBiFunction.one((user, date) -> {
                                    user.setLastPayout(date);
                                    return null;
                                })
                        )
                        .withRequiredProperties(List.of(User.LAST_PAYOUT, User.CREATE_DATE))
                        .withAvailableFunction(sameAs(apiFieldName))
                        .withEditableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.LAST_PAYOUT),
                ApiField.<User, LocalDateTime>convertibleApiField(LocalDateTime.class, User.CREATE_DATE)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.CREATE_DATE),
                getAdfoxInfoApiField(apiFieldName),
                ApiField.<User, Long>convertibleApiField(Long.class, User.COMMON_OFFER)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.COMMON_OFFER),
                ApiField.<User, Long>convertibleApiField(Long.class, User.NOTIFICATIONS_COUNT)
                        .withAvailableFunction(sameAs(apiFieldName))
                        .build(UserCrnkJsonFieldAliases.NOTIFICATIONS_COUNT),
                ApiField.<User, Integer>convertibleApiField(Integer.class, User.BUSINESS_RULES_COUNT)
                        .withAvailableFunction(sameAs(apiFieldName))
                        .build(UserCrnkJsonFieldAliases.BUSINESS_RULES_COUNT),
                ApiField.<User, Integer>convertibleApiField(Integer.class, User.BUSINESS_RULES_ACTIVE_COUNT)
                        .withAvailableFunction(sameAs(apiFieldName))
                        .build(UserCrnkJsonFieldAliases.BUSINESS_RULES_ACTIVE_COUNT),
                getActiveContractApiField(apiFieldName),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.HAS_MOBILE_MEDIATION)
                        .withAvailableFunction(selfViewOrCanViewAllFieldsAndHasRightConsumer(
                                UserRightsEnum.VIEW_FIELD__HAS_MOBILE_MEDIATION))
                        .withEditableFunction(sameAs(apiFieldName))
                        .build(UserCrnkJsonFieldAliases.HAS_MOBILE_MEDIATION),
                getHasRsya(apiFieldName),
                getHasTutbyAgreement(),
                getAdfoxApiField(apiFieldName),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.PAID_OFFER)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.PAID_OFFER),
                ApiField.<User, String>convertibleApiField(String.class, User.INN)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.INN),
                ApiField.<User, String>convertibleApiField(String.class, User.COOPERATION_FORM)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.COOPERATION_FORM),
                ApiField.<User, Integer>convertibleApiField(Integer.class, User.SELF_EMPLOYED)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.SELF_EMPLOYED),
                ApiField.<User, List<Long>>readableApiField(ModelClassUtils.getListClass(Long.class),
                                User.MODERATION_REASON)
                        .withAvailableFunction(checkableByRight(UserRightsEnum.VIEW_FIELD__MODERATION_REASON))
                        .withEditableFunction(checkableByRight(UserRightsEnum.EDIT_FIELD__MODERATION_REASON))
                        .withDefaultsFunction(paramsContext -> {
                            var user = paramsContext.getUserAuthentication();

                            return Map.of(
                                    UserCrnkJsonFieldAliases.MODERATION_REASON,
                                    moderationReasonDictionaryValuesProvider.getDictionaryValues(user)
                            );
                        })
                        .build(UserCrnkJsonFieldAliases.MODERATION_REASON),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.HAS_GAME_OFFER)
                        .withAvailableFunction(sameAs(apiFieldName))
                        .withEditableFunction(checkableByRight(UserRightsEnum.EDIT_FIELD__HAS_GAME_OFFER))
                        .build(UserCrnkJsonFieldAliases.HAS_GAME_OFFER),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.IS_DELETED)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.IS_DELETED),
                ApiField.<User, LocalDateTime>convertibleApiField(LocalDateTime.class, User.DELETION_DATE)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.DELETION_DATE),
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.IS_FORM_DONE)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.IS_FORM_DONE),
                getModerationReason(),
                getContractorType(apiFieldName));
    }

    private ApiField<User> getExcludedDomains() {
        return ApiField.<User, List<String>>convertibleApiField(ModelClassUtils.getListClass(String.class),
                        User.EXCLUDED_DOMAINS)
                .withAvailableFunction(checkableAnd(List.of(
                        selfViewOrCanViewAllFields(), getSiteAndPartnerAccessFunction())))
                .withEditableFunction(checkable(getSiteAndPartnerEditableFunction()))
                .build(UserCrnkJsonFieldAliases.EXCLUDED_DOMAINS);
    }

    private ApiField<User> getName() {
        return ApiField.<User, String>convertibleApiField(String.class, User.NAME)
                .withAvailableFunction(checkable(selfViewOrCanViewAllFields()))
                .withEditableFunction(checkable(selfEditOrCanEditAllFields()))
                .build(UserCrnkJsonFieldAliases.NAME);
    }

    private ApiField<User> getContractorType(ApiField<User> apiFieldName) {
        return ApiField.readableApiField(String.class, User.CONTRACTOR_TYPE,
                        BatchFunction.<User, String>one(user -> fillContractorType(user,
                                authenticationFacade.getUserAuthentication())))
                .withAvailableFunction(sameAs(apiFieldName))
                .build(UserCrnkJsonFieldAliases.CONTRACTOR_TYPE);
    }

    private String fillContractorType(User user, UserAuthentication ua) {
        var selfemployed = user.getSelfEmployed();
        var cooperationForm = user.getCooperationForm();
        if (selfemployed != null && selfemployed == CoreConstants.SELF_EMPLOYED_STATUS_READY) {
            return messageSource.getMessage(UserMsg.SELFEMPLOYED);
        }

        if (cooperationForm != null) {
            Matcher urMatcher = UR_TYPE_PATTERN.matcher(cooperationForm);
            if (urMatcher.find()) {
                var inn = user.getInn();
                if (inn != null && inn.length() == 12) {
                    return messageSource.getMessage(UserMsg.INDIVIDUAL_ENTREPRENEUR);
                } else {
                    return messageSource.getMessage(UserMsg.ENTITY);
                }
            } else {
                Matcher phMatcher = PH_TYPE_PATTERN.matcher(cooperationForm);
                if (phMatcher.find()) {
                    return messageSource.getMessage(UserMsg.INDIVIDUAL);
                }
            }
        }
        return null;
    }

    private ApiField<User> getModerationReason() {
        return ApiField.readableApiField(String.class, User.MODERATION_REASON,
                        BatchFunction.<User, String>one(user -> fillModerationReasonNames(user,
                                authenticationFacade.getUserAuthentication())))
                .withAvailableFunction(checkable(selfViewOrCanViewAllFields()))
                .build(UserCrnkJsonFieldAliases.MODERATION_REASON_NAME);
    }

    private ApiField<User> getAdfoxApiField(ApiField<User> apiFieldName) {
        return ApiField
                .<User, Boolean>convertibleApiField(Boolean.class, User.ADFOX_OFFER)
                .withAvailableFunction(sameAs(apiFieldName))
                .withEditableFunction(checkableOr(List.of(
                        new ApiFieldsAccessRulesFunction<>(
                                (userAuthentication, model) ->
                                        userAuthentication.userHasRight(UserRightsEnum.EDIT_FIELD__ADFOX_OFFER)),
                        ApiFieldsAccessRulesFunction.and(List.of(
                                selfEditOrCanEditAllFields(),
                                new ApiFieldsAccessRulesFunction<>(
                                        (userAuthentication, editableUser) -> {
                                            var user = editableUser.getPatchedOrElseActual();
                                            return user.getPaidOffer() != null
                                                    && user.getPaidOffer()
                                                    && user.getHasCommonOffer() != null
                                                    && user.getHasCommonOffer()
                                                    && user.getAdfoxInfos() != null;
                                        },
                                        List.of(User.PAID_OFFER, User.HAS_COMMON_OFFER, User.ADFOX_INFOS)
                                )
                        ))
                )))
                .build(UserCrnkJsonFieldAliases.ADFOX_OFFER);
    }

    private ApiField<User> getHasTutbyAgreement() {
        return ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.HAS_TUTBY_AGREEMENT)
                .withAvailableFunction(selfViewOrCanViewAllFieldsAndHasRightConsumer(
                        UserRightsEnum.VIEW_FIELD__HAS_TUTBY_AGREEMENT))
                .withEditableFunction(checkable(new ApiFieldsAccessRulesFunction<>(
                        (userAuthentication, editableUser) -> {
                            var user = editableUser.getPatchedOrElseActual();
                            return userAuthentication.userHasRight(UserRightsEnum.EDIT_FIELD__HAS_TUTBY_AGREEMENT)
                                    && user.getIsTutby() != null
                                    && user.getIsTutby();
                        }, User.IS_TUTBY))
                )
                .build(UserCrnkJsonFieldAliases.HAS_TUTBY_AGREEMENT);
    }

    private ApiField<User> getHasRsya(ApiField<User> apiFieldName) {
        return ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.HAS_RSYA)
                .withAvailableFunction(checkableOr(List.of(
                        new ApiFieldsAccessRulesFunction<>(
                                (userAuthentication, user) -> Boolean.TRUE.equals(user.getIsMobileMediation()),
                                User.IS_MOBILE_MEDIATION),
                        selfViewOrCanViewAllFieldsAndHasRight(UserRightsEnum.VIEW_FIELD__HAS_RSYA))))
                .withEditableFunction(sameAs(apiFieldName))
                .build(UserCrnkJsonFieldAliases.HAS_RSYA);
    }

    private ApiField<User> getActiveContractApiField(ApiField<User> apiFieldName) {
        return ApiField
                .readableApiField(ActiveContractCrnk.class,
                        User.CLIENT_ID,
                        BatchFunction.<User, ActiveContractCrnk>many(users -> {
                            var clientIds = users.stream()
                                    .map(user -> Map.entry(user.getClientId(), user.getLogin()))
                                    .distinct()
                                    .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
                            var crnkContractsStream = balanceDocumentService.getActiveContracts(clientIds.keySet())
                                    .stream()
                                    .map(activeContractCrnkMapper::activeContractToActiveContractCrnk);

                            if (authenticationFacade.getUserAuthentication()
                                    .userHasRight(UserRightsEnum.PARTNER_ACTS_VIEW_FILTER__LOGIN)) {
                                crnkContractsStream = crnkContractsStream.peek(activeContractCrnk -> {
                                    if (activeContractCrnk != null && activeContractCrnk.getPerson() != null) {
                                        activeContractCrnk.getPerson().setLogin(
                                                clientIds.get(activeContractCrnk.getPerson().getClientId())
                                        );
                                    }
                                });
                            }

                            return crnkContractsStream.collect(Collectors.toList());
                        }))
                .withAvailableFunction(sameAs(apiFieldName))
                .addCheckInnerField(UserRightsEnum.VIEW_ACTIVE_CONTRACT_REWARD_TYPE, "Contract", "reward_type")
                .addCheckInnerField(UserRightsEnum.VIEW_FIELD__CLENT_ID, "Person", "client_id")
                .addCheckInnerField(UserRightsEnum.VIEW_FIELD__ID, "Person", "id")
                .addCheckInnerField(UserRightsEnum.PARTNER_ACTS_VIEW_FILTER__LOGIN, "Person", "login")
                .build(UserCrnkJsonFieldAliases.ACTIVE_CONTRACT);
    }

    private ApiField<User> getRolesApiField() {
        return ApiField
                .convertibleApiField(ModelClassUtils.getSetClass(RoleCrnk.class), ROLES,
                        BatchFunction.one(user -> user.getRoles().stream().map(
                                roleCrnkMapper::roleToRoleCrnk).collect(Collectors.toSet())),
                        BatchBiFunction.<User, Set<RoleCrnk>>one((user, roleCrnks) -> {
                            user.setRoles(roleCrnks.stream().map(roleCrnkMapper::roleCrnkToRole)
                                    .collect(Collectors.toSet()));
                            return null;
                        }))
                .withAvailableFunction(selfViewOrCanViewAllFieldsAndHasRightConsumer(
                        UserRightsEnum.VIEW_FIELD__ROLES))
                .withEditableFunction(
                        checkable(new ApiFieldsAccessRulesFunction<>(
                                        (userAuthentication, user) ->
                                                userAuthentication.userHasRight(
                                                        UserRightsEnum.DO_USER_ACTION_SET_USER_ROLE) &&
                                                        userAuthentication.userHasRight(
                                                                UserRightsEnum.DO_USER_ACTION_REVOKE_ROLES)
                                )
                        ))
                .build(UserCrnkJsonFieldAliases.ROLES);
    }

    private ApiField<User> getAdfoxInfoApiField(ApiField<User> apiFieldName) {
        return ApiField.convertibleApiField(
                        ModelClassUtils.getListClass(AdfoxInfoCrnk.class),
                        User.ADFOX_INFOS,
                        BatchFunction.one(user -> Objects.isNull(user.getAdfoxInfos())
                                ? null
                                : user.getAdfoxInfos().stream()
                                .map(adfoxInfoCrnkMapper::adfoxInfoToAdfoxInfoCrnk).collect(Collectors.toList())),
                        BatchBiFunction.<User, List<AdfoxInfoCrnk>>one((user, adfoxInfosCrnk) -> {
                            user.setAdfoxInfos(adfoxInfosCrnk.stream()
                                    .map(adfoxInfoCrnkMapper::adfoxInfoCrnkToAdfoxInfo)
                                    .collect(Collectors.toList()));
                            return null;
                        }))
                .withAvailableFunction(sameAs(apiFieldName))
                .build(UserCrnkJsonFieldAliases.ADFOX_INFO);
    }

    private List<ApiField<User>> getPayoneerFields(ApiField<User> apiFieldName) {
        return List.of(
                ApiField.<User, Boolean>convertibleApiField(Boolean.class, User.PAYONEER)
                        .withAvailableFunction(checkable(selfViewOrCanViewAllFields()))
                        .build(UserCrnkJsonFieldAliases.PAYONEER),
                ApiField.<User, String>convertibleApiField(String.class, User.PAYONEER_CURRENCY)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.PAYONEER_CURRENCY),
                ApiField.<User, String>convertibleApiField(String.class, User.PAYONEER_PAYEE_ID)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.PAYONEER_PAYEE_ID),
                ApiField.<User, String>convertibleApiField(String.class, User.PAYONEER_URL)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.PAYONEER_URL),
                ApiField.<User, Integer>convertibleApiField(Integer.class, User.PAYONEER_STEP)
                        .withAvailableFunction(sameAs(apiFieldName)).build(UserCrnkJsonFieldAliases.PAYONEER_STEP)
        );
    }

    private List<ApiField<User>> getCurrencyFields() {
        var currentCurrencyField = ApiField.readableApiField(String.class,
                        User.CURRENT_CURRENCY,
                        BatchFunction.<User, String>one(this::getUserCurrentCurrency))
                .withAvailableFunction(checkable(selfViewOrCanViewAllFields()))
                .build(UserCrnkJsonFieldAliases.CURRENT_CURRENCY);

        var nextCurrencyField = ApiField.convertibleApiField(String.class, User.NEXT_CURRENCY,
                        BatchFunction.<User, String>one(user ->
                                Objects.requireNonNullElse(user.getNextCurrency(), getUserCurrentCurrency(user))
                        ),
                        BatchBiFunction.one((user, currency) -> {
                            User.NEXT_CURRENCY.set(user, currency);
                            return null;
                        }))
                .withRequiredProperties(User.CURRENT_CURRENCY)
                .withAvailableFunction(sameAs(currentCurrencyField))
                .withEditableFunction(checkable(ApiFieldsAccessRulesFunction.and(List.of(
                        selfEditOrCanEditAllFields(),
                        getNextCurrencyEditableFunction()
                ))))
                .build(UserCrnkJsonFieldAliases.NEXT_CURRENCY);
        return List.of(
                currentCurrencyField,
                nextCurrencyField,
                ApiField.<User, BigDecimal>convertibleApiField(BigDecimal.class, User.CURRENCY_RATE)
                        .withAvailableFunction(sameAs(currentCurrencyField))
                        .withEditableFunction(sameAs(nextCurrencyField))
                        .build(UserCrnkJsonFieldAliases.CURRENCY_RATE)
        );
    }

    @Override
    public Map<String, CrnkFilter<User, ?>> getFilters() {
        CheckRightsCrnkFilterExposeStrategy.Factory expose
                = new CheckRightsCrnkFilterExposeStrategy.Factory(authenticationFacade);

        List<CrnkFilter<User, ?>> filtersList = List.of(
                CrnkFilter.builder(FilterNameEnum.ID)
                        .toCrnkFilter(UserFilters.ID),
                CrnkFilter.builder(FilterNameEnum.UID)
                        .toCrnkFilter(UserFilters.UID),
                CrnkFilter.builder(FilterNameEnum.IS_DELETED)
                        .toCrnkFilter(UserFilters.IS_DELETED),
                CrnkFilter.builder(FilterNameEnum.DELETION_DATE)
                        .toCrnkFilter(UserFilters.DELETION_DATE),

                CrnkFilter.builder(FilterNameEnum.CLIENT_ID)
                        .setLabel(UserMsg.CLIENT_ID)
                        .setHint(UserMsg.CLIENT_ID_HINT)
                        .setExposed(expose.forRights(UserRightsEnum.VIEW, UserRightsEnum.VIEW_FIELD__CLENT_ID))
                        .toCrnkFilter(UserFilters.CLIENT_ID),

                CrnkFilter.builder(FilterNameEnum.LOGIN)
                        .setLabel(UserMsg.LOGIN)
                        .setExposed(expose.forRights(UserRightsEnum.VIEW))
                        .toCrnkFilter(UserFilters.LOGIN),

                CrnkFilter.builder(FilterNameEnum.NAME)
                        .setLabel(UserMsg.NAME)
                        .setExposed(expose.forRights(UserRightsEnum.VIEW))
                        .toCrnkFilter(UserFilters.NAME),

                CrnkFilter.builder(FilterNameEnum.LASTNAME)
                        .setLabel(UserMsg.LAST_NAME)
                        .setExposed(expose.forRights(UserRightsEnum.VIEW))
                        .toCrnkFilter(UserFilters.LASTNAME),

                CrnkFilter.builder(FilterNameEnum.EMAIL)
                        .setLabel(PersonMsg.EMAIL)
                        .setExposed(expose.forRights(UserRightsEnum.VIEW))
                        .toCrnkFilter(UserFilters.EMAIL),

                CrnkFilter.builder(FilterNameEnum.PHONE)
                        .setLabel(PersonMsg.PHONE)
                        .setExposed(expose.forRights(UserRightsEnum.VIEW))
                        .toCrnkFilter(UserFilters.PHONE),

                CrnkFilter.builder(FilterNameEnum.ACCOUNTANT_EMAIL)
                        .setLabel(PersonMsg.ACCOUNTANT_EMAIL)
                        .toCrnkFilter(UserFilters.ACCOUNTANT_EMAIL),

                CrnkFilter.builder(FilterNameEnum.DOMAIN_LOGIN)
                        .setLabel(UserMsg.DOMAIN_LOGIN)
                        .toCrnkFilter(UserFilters.DOMAIN_LOGIN),

                CrnkFilter.builder(FilterNameEnum.NEWSLETTER)
                        .toCrnkFilter(UserFilters.NEWSLETTER),

                CrnkFilter.builder(FilterNameEnum.IS_TUTBY)
                        .toCrnkFilter(UserFilters.IS_TUTBY),

                CrnkFilter.builder(FilterNameEnum.HAS_TUTBY_AGREEMENT)
                        .toCrnkFilter(UserFilters.HAS_TUTBY_AGREEMENT),

                CrnkFilter.builder(FilterNameEnum.HAS_COMMON_OFFER)
                        .toCrnkFilter(UserFilters.HAS_COMMON_OFFER),

                CrnkFilter.builder(FilterNameEnum.BUSINESS_UNIT)
                        .toCrnkFilter(UserFilters.BUSINESS_UNIT),
                CrnkFilter.builder(FilterNameEnum.MULTISTATE)
                        .toCrnkFilter(
                                UserFilters.MULTISTATE.asLongFilter(),
                                new MultistateCrnkFilterValueParser<>(userMultistateGraph,
                                        userMultistateExpressionParser)
                        ),
                CrnkFilter.builder(FilterNameEnum.ROLE_ID)
                        .setLabel(UserMsg.ROLE)
                        .setExposed(expose.forRights(UserRightsEnum.VIEW, UserRightsEnum.VIEW_FIELD__ROLES))
                        .toCrnkFilter(UserFilters.ROLE_ID, rolesDictionaryValuesProvider),
                CrnkFilter.builder(FilterNameEnum.NO_STAT_MONITORING_EMAILS)
                        .toCrnkFilter(UserFilters.NO_STAT_MONITORING_EMAILS),
                CrnkFilter.builder(FilterNameEnum.NEED_TO_EMAIL_PROCESSING)
                        .toCrnkFilter(UserFilters.NEED_TO_EMAIL_PROCESSING),
                CrnkFilter.builder(FilterNameEnum.IS_ADFOX_PARTNER)
                        .toCrnkFilter(UserFilters.IS_ADFOX_PARTNER),
                CrnkFilter.builder(FilterNameEnum.ADFOX_ID)
                        .setLabel(UserMsg.ADFOX_ID)
                        .toCrnkFilter(UserFilters.ADFOX_ID),
                CrnkFilter.builder(FilterNameEnum.IS_YANDEX)
                        .toCrnkFilter(UserFilters.IS_YANDEX_FILTER),
                CrnkFilter.builder(FilterNameEnum.HAS_BUSINESS_RULE)
                        .setLabel(UserMsg.HAS_BUSINESS_RULES)
                        .toCrnkFilter(UserFilters.HAS_BUSINESS_RULE),
                CrnkFilter.builder(FilterNameEnum.CONTRACT_ID)
                        .setLabel(UserMsg.CONTRACT_ID)
                        .setExposed(expose.forRights(UserRightsEnum.VIEW, UserRightsEnum.VIEW_FIELD__CONTRACT_ID))
                        .toCrnkFilter(
                                UserFilters.CLIENT_ID,
                                new ContractNumberCrnkFilterValueParser(balanceDocumentService)
                        ),
                CrnkFilter.builder(FilterNameEnum.USER_TYPE)
                        .setLabel(UserMsg.USER_TYPE)
                        .setExposed(expose.forRights(UserRightsEnum.VIEW,
                                UserRightsEnum.VIEW_SEARCH_FILTERS__USER_TYPE))
                        .toCrnkFilter(
                                UserFilters.USER_TYPE,
                                new UserTypeCrnkFilterValueParser(authenticationFacade),
                                userTypeDictionaryValuesProvider
                        ),
                CrnkFilter.builder(FilterNameEnum.MODERATION_REASON)
                        .setLabel(ModerationReasonMsg.MODERATION_REASON)
                        .setExposed(expose.forRights(UserRightsEnum.VIEW_FIELD__MODERATION_REASON))
                        .toCrnkFilter(UserFilters.MODERATION_REASON_FILTER,
                                moderationReasonDictionaryValuesProvider)
        );

        return filtersList.stream()
                .collect(Collectors.toMap(CrnkFilter::getName, Function.identity()));
    }

    private String fillModerationReasonNames(User user, UserAuthentication ua) {
        if (user.getModerationReason() == null) {
            return "";
        }
        Map<Long, List<Long>> moderationReasonIdsMap = Map.of(user.getId(), user.getModerationReason());

        String language = LocaleContextHolder.getLocale().getLanguage();
        boolean isManager = ua.userHasRight(MODERATION_REASON_MANAGER_TXT_VIEW);

        return moderationReasonService
                .renderModerationReason(moderationReasonIdsMap, language, isManager).get(user.getId());

    }

    private ApiFieldsAccessRulesFunction<User> getSiteAndPartnerAccessFunction() {
        return new ApiFieldsAccessRulesFunction<>(
                (userAuthentication, user) -> checkSiteAndMobileRights(user),
                ROLES);
    }

    private ApiFieldsAccessRulesFunction<EditableData<User>> getSiteAndPartnerEditableFunction() {
        return new ApiFieldsAccessRulesFunction<>(
                (userAuthentication, user) ->
                        (userAuthentication.userHasRight(UserRightsEnum.DO_CONTEXT_ON_SITE_ADD) ||
                                userAuthentication.userHasRight(UserRightsEnum.DO_SEARCH_ON_SITE_ADD) ||
                                userAuthentication.userHasRight(UserRightsEnum.DO_MOBILE_APP_ADD))
                                && checkSiteAndMobileRights(user.getPatchedOrElseActual()),
                ROLES);

    }

    private Boolean checkSiteAndMobileRights(User user) {
        return user.getRoles() != null
                && user.getRoles().stream().anyMatch(
                role -> role.getRoleId() == RoleSet.SITE_PARTNER.getRoleId()
                        || role.getRoleId() == RoleSet.MOBILE_PARTNER.getRoleId());
    }

    private ApiFieldsAccessRulesFunction<EditableData<User>> getNextCurrencyEditableFunction() {
        return new ApiFieldsAccessRulesFunction<>((userAuthentication, editableUser) -> {
            if (!userAuthentication.userHasFeature(UserFeatureEnum.CPM_CURRENCY)) {
                return false;
            }
            var user = editableUser.getActual();
            return user.getNextCurrency() == null || user.getNextCurrency().equals(getUserCurrentCurrency(user));
        },
                User.CURRENT_CURRENCY);
    }

    private String getUserCurrentCurrency(User user) {
        return Objects.requireNonNullElse(user.getCurrentCurrency(), CoreConstants.Currency.DEFAULT);
    }
}
