package ru.yandex.partner.jsonapi.response;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import io.crnk.core.engine.document.ErrorData;
import io.crnk.core.engine.document.ErrorDataBuilder;
import io.crnk.core.engine.error.ErrorResponse;
import io.crnk.core.exception.CrnkMappableException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.context.MessageSourceAware;
import org.springframework.context.support.DelegatingMessageSource;
import org.springframework.context.support.MessageSourceAccessor;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.web.server.ResponseStatusException;

import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.messages.JsonapiErrorMsg;
import ru.yandex.partner.libs.auth.exception.ExceptionWithMeta;
import ru.yandex.partner.libs.exceptions.I18nResponseStatusException;
import ru.yandex.partner.libs.exceptions.Translatable;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.i18n.TranslatableError;

@Component
public class PartnerErrorResponseBuilder implements MessageSourceAware {

    private final ExceptionToErrorIdMapper exceptionToErrorIdMapper;
    protected MessageSourceAccessor messages = new MessageSourceAccessor(new DelegatingMessageSource());

    @Autowired
    public PartnerErrorResponseBuilder(ExceptionToErrorIdMapper exceptionToErrorIdMapper) {
        this.exceptionToErrorIdMapper = exceptionToErrorIdMapper;
    }

    @Override
    public void setMessageSource(@Nonnull MessageSource messageSource) {
        this.messages = new MessageSourceAccessor(messageSource);
    }

    public ErrorResponse buildErrorResponse(CrnkMappableException exception) {
        var httpStatus = HttpStatus.valueOf(exception.getHttpStatus());
        Integer id = calculateId(httpStatus);
        return buildErrorResponse(
                id,
                httpStatus,
                exception.getErrorData().getDetail(),
                exception
        );
    }

    public ErrorResponse buildErrorResponse(ResponseStatusException exception) {
        String detail = responseExceptionDetail(exception);
        Integer id = calculateId(exception.getStatus());
        return buildErrorResponse(
                id,
                exception.getStatus(),
                detail,
                exception
        );
    }

    public ErrorResponse buildErrorResponse(I18nResponseStatusException exception) {
        String detail = responseExceptionDetail(exception);
        return buildErrorResponse(
                exception.getHttpErrorStatus().getId(),
                exception.getStatus(),
                detail,
                exception
        );
    }

    public ErrorResponse buildErrorResponse(CrnkResponseStatusException exception) {
        List<TranslatableError> apiErrors = exception.getApiErrors();

        List<ErrorData> errors = new ArrayList<>(apiErrors.size());

        HttpStatus httpStatus = exception.getHttpErrorStatusEnum().getHttpStatus();

        Integer id = exception.getHttpErrorStatusEnum().getId();
        String code = calculateCode(httpStatus, id);
        String title = calculateTitle(httpStatus, id);

        errors.addAll(apiErrors.stream().map(apiError -> new ErrorDataBuilder()
                .setId(String.valueOf(id))
                .setCode(code)
                .setTitle(title)
                .setSourcePointer(apiError.getPath())
                .setDetail(apiError.getMessage() == null ? null : messages.getMessage(apiError.getMessage()))
                .build()
        ).collect(Collectors.toList()));

        return ErrorResponse.builder()
                .setStatus(httpStatus.value())
                .setErrorData(errors)
                .build();
    }

    public ErrorResponse buildErrorResponse(Integer id, HttpStatus httpStatus, String detail, Exception exception) {
        ErrorDataBuilder errorDataBuilder = new ErrorDataBuilder()
                .setId(String.valueOf(id))
                .setCode(calculateCode(httpStatus, id))
                .setTitle(calculateTitle(httpStatus, id))
                .setDetail(detail);

        Throwable cause = exception.getCause();
        if (cause instanceof ExceptionWithMeta) {
            errorDataBuilder.setMeta(((ExceptionWithMeta) cause).getMeta());
        }

        return ErrorResponse.builder()
                .setStatus(httpStatus.value())
                .setSingleErrorData(errorDataBuilder.build())
                .build();
    }

    private String responseExceptionDetail(I18nResponseStatusException exception) {
        GettextMsg msg = exception.getI18nMessage();
        if (msg != null) {
            return messages.getMessage(msg);
        }

        return responseExceptionDetail(exception.getCause());
    }

    private String responseExceptionDetail(ResponseStatusException exception) {
        String reason = exception.getReason();
        if (reason != null && !exception.getStatus().getReasonPhrase().equals(reason)) {
            return reason;
        }

        return responseExceptionDetail(exception.getCause());
    }

    private String responseExceptionDetail(@Nullable Throwable exception) {
        if (exception == null) {
            return null;
        }

        if (exception instanceof Translatable) {
            GettextMsg msg = ((Translatable) exception).getI18nMessage();
            return msg == null ? null : messages.getMessage(msg);
        } else {
            return exception.getMessage();
        }
    }

    private Integer calculateId(HttpStatus httpStatus) {
        return exceptionToErrorIdMapper.mapHttpStatusToErrorId(httpStatus.value());
    }


    private String calculateTitle(HttpStatus httpStatus, Integer errorId) {
        JsonapiErrorMsg titleMsg = Optional.ofNullable(JsonapiErrorMsg.forErrorId(errorId))
                .orElseGet(() -> JsonapiErrorMsg.forStatus(httpStatus));
        return titleMsg == null ? httpStatus.getReasonPhrase()
                : messages.getMessage(titleMsg);
    }

    private String calculateCode(HttpStatus httpStatus, Integer errorId) {
        // Не добавляем код для ошибки 404 по аналогии с перлом
        // Возможно, есть смысл исправить в перле и убрать этот костыль
        // Также не пишем статус для ошибок валидации (id ==4)
        if (httpStatus == HttpStatus.NOT_FOUND || errorId == 4) {
            return null;
        } else {
            return String.valueOf(httpStatus.value());
        }
    }
}
