package ru.yandex.partner.jsonapi.response;

import java.io.IOException;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.crnk.core.engine.dispatcher.Response;
import io.crnk.core.engine.error.ErrorResponse;
import io.crnk.core.exception.CrnkMappableException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.server.ResponseStatusException;
import org.springframework.web.util.NestedServletException;

import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.messages.JsonapiErrorMsg;
import ru.yandex.partner.libs.auth.filter.ExceptionResponseWriter;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;
import ru.yandex.partner.libs.exceptions.I18nResponseStatusException;

public class PartnerExceptionResponseWriter implements ExceptionResponseWriter {

    private static final Logger LOGGER = LoggerFactory.getLogger(PartnerExceptionResponseWriter.class);

    private static final String CONTENT_TYPE = "application/vnd.api+json; charset=utf-8";
    private final PartnerErrorResponseBuilder partnerErrorResponseBuilder;
    private final ObjectMapper objectMapper;

    public PartnerExceptionResponseWriter(ObjectMapper objectMapper,
                                          PartnerErrorResponseBuilder partnerErrorResponseBuilder) {
        this.objectMapper = objectMapper;
        this.partnerErrorResponseBuilder = partnerErrorResponseBuilder;
    }

    @Override
    public void writeExceptionResponse(HttpServletRequest httpRequest, HttpServletResponse httpResponse,
                                       Throwable exception) throws IOException {
        Response errorResponse = toResponse(exception);

        Map<String, String> headers = httpResponse.getHeaderNames().stream()
                .collect(Collectors.toMap(Function.identity(), httpResponse::getHeader));

        httpResponse.reset();

        headers.forEach(httpResponse::setHeader);
        httpResponse.setStatus(errorResponse.getHttpStatus());
        httpResponse.setContentType(CONTENT_TYPE);
        httpResponse.getWriter().println(objectMapper.writeValueAsString(errorResponse.getDocument()));
    }

    public Response toResponse(Throwable exception) {
        if (exception instanceof NestedServletException && exception.getCause() != null) {
            exception = exception.getCause();
        }

        ErrorResponse errorResponse;
        if (exception instanceof I18nResponseStatusException) {
            errorResponse = partnerErrorResponseBuilder.buildErrorResponse((I18nResponseStatusException) exception);
        } else if (exception instanceof ResponseStatusException) {
            errorResponse = partnerErrorResponseBuilder.buildErrorResponse((ResponseStatusException) exception);
        } else if (exception instanceof CrnkResponseStatusException) {
            errorResponse = partnerErrorResponseBuilder.buildErrorResponse((CrnkResponseStatusException) exception);
        } else if (exception instanceof CrnkMappableException) {
            errorResponse = partnerErrorResponseBuilder.buildErrorResponse((CrnkMappableException) exception);
        } else {
            LOGGER.error(exception.getMessage(), exception);
            errorResponse = partnerErrorResponseBuilder.buildErrorResponse(
                    new I18nResponseStatusException(HttpErrorStatusEnum.ERROR__INTERNAL,
                            JsonapiErrorMsg.INTERNAL_SERVER_ERROR)
            );
        }

        return errorResponse.toResponse();
    }
}
