package ru.yandex.partner.jsonapi.service;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.collect.Maps;
import io.crnk.core.engine.document.Document;
import io.crnk.core.engine.document.Relationship;
import io.crnk.core.engine.document.Resource;
import io.crnk.core.engine.document.ResourceIdentifier;
import io.crnk.core.engine.http.HttpRequestContext;
import io.crnk.core.queryspec.AbstractPathSpec;
import io.crnk.core.queryspec.QuerySpec;
import io.crnk.core.resource.links.LinksInformation;
import io.crnk.core.resource.meta.MetaInformation;
import io.crnk.core.utils.Nullable;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.partner.core.utils.OrderBy;
import ru.yandex.partner.jsonapi.crnk.BaseCrnkLinksInformation;
import ru.yandex.partner.jsonapi.crnk.BaseCrnkMetaInformation;
import ru.yandex.partner.jsonapi.crnk.DynamicResourceRepository;
import ru.yandex.partner.jsonapi.crnk.QuerySpecUtil;
import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.IncomingApiFields;
import ru.yandex.partner.jsonapi.crnk.fields.QueryParamsContext;
import ru.yandex.partner.jsonapi.crnk.filter.description.FilterDescriptionService;
import ru.yandex.partner.jsonapi.crnk.filter.parser.FilterNode;
import ru.yandex.partner.jsonapi.messages.CommonMsg;
import ru.yandex.partner.jsonapi.messages.JsonapiErrorMsg;
import ru.yandex.partner.jsonapi.models.ApiModel;
import ru.yandex.partner.jsonapi.models.ApiService;
import ru.yandex.partner.jsonapi.models.DependsHolder;
import ru.yandex.partner.jsonapi.models.relationships.ApiRelationship;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;
import ru.yandex.partner.libs.i18n.MsgWithArgs;

@Service
public class CrnkAdditionalDataService {

    private final ObjectMapper objectMapper;
    private final AuthenticationFacade authenticationFacade;
    private final FilterDescriptionService filterDescriptionService;

    public CrnkAdditionalDataService(ObjectMapper objectMapper,
                                     AuthenticationFacade authenticationFacade,
                                     FilterDescriptionService filterDescriptionService) {
        this.objectMapper = objectMapper;
        this.authenticationFacade = authenticationFacade;
        this.filterDescriptionService = filterDescriptionService;
    }

    public <M extends ModelWithId> Document getDepends(HttpRequestContext httpRequestContext, ApiModel<M> apiModel) {
        var response = new Document();
        var resource = new Resource();
        resource.setId(apiModel.getResourceType());
        resource.setType(apiModel.getResourceType() + "_depends");
        var depends =
                apiModel.getDepends().orElseThrow(() -> new CrnkResponseStatusException(
                        HttpErrorStatusEnum.ERROR__NOT_FOUND,
                        JsonapiErrorMsg.NOT_FOUND));
        ObjectNode objectNode = objectMapper.valueToTree(depends);
        var dependsMap = new HashMap<String, JsonNode>();
        for (var it = objectNode.fields(); it.hasNext(); ) {
            var field = it.next();
            dependsMap.put(field.getKey(), field.getValue());
        }
        resource.setAttributes(dependsMap);
        response.setData(Nullable.of(resource));
        var linksInformation = new BaseCrnkLinksInformation();
        linksInformation.setSelf(httpRequestContext.getBaseUrl() + "/" + apiModel.getResourceType() + "/depends");
        response.setLinks(objectMapper.valueToTree(linksInformation));
        return response;
    }

    public <M extends ModelWithId> Document getAddFields(HttpRequestContext httpRequestContext, String attributes,
                                                         DynamicResourceRepository<M> dynamicResourceRepository) {
        M model;
        if (Objects.nonNull(attributes)) {
            try {
                var attr = objectMapper.readValue(attributes, new TypeReference<Map<String, JsonNode>>() {
                });
                var tempResource = new Resource();
                tempResource.setAttributes(attr);
                var updatedModelProperties =
                        new IncomingApiFields<>(
                                Maps.<String, ApiField<M>>newHashMapWithExpectedSize(attr.size()));

                var errorParsing = new LinkedList<DefectInfo<Defect>>();
                model = dynamicResourceRepository.parseModel(tempResource, updatedModelProperties, errorParsing);

            } catch (JsonProcessingException e) {
                throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS, CommonMsg.INCORRECT_JSON, e);
            }
        } else {
            model = dynamicResourceRepository.getApiService().getApiModel().createNewInstance();
        }

        var apiAddFields =
                dynamicResourceRepository.getApiService().getApiFieldsService().getAddFields(authenticationFacade,
                        model);

        var resources = StreamEx.of(apiAddFields).map(it -> {
            Resource resource = new Resource();
            resource.setId(it);
            resource.setType(dynamicResourceRepository.getResourceType() + "_add_fields");
            return resource;
        }).sorted(Comparator.comparing(ResourceIdentifier::getId)).toList();

        var response = new Document();
        var linksInformation = new BaseCrnkLinksInformation();
        String link = addFieldsLink(dynamicResourceRepository.getResourceType(),
                httpRequestContext, dynamicResourceRepository.getApiService());
        if (link == null) {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_FOUND, JsonapiErrorMsg.NOT_FOUND);
        }
        linksInformation.setSelf(link);
        var metaInformation = new BaseCrnkMetaInformation();
        dynamicResourceRepository.getApiService().fillMetaInformation(metaInformation);
        metaInformation.setCount((long) resources.size());
        fillRightMetaInformation(metaInformation, dynamicResourceRepository.getApiService());

        response.setData(Nullable.of(resources));
        response.setLinks(objectMapper.valueToTree(linksInformation));
        response.setMeta(objectMapper.valueToTree(metaInformation));

        return response;
    }

    public <M extends ModelWithId> Document getDefaults(HttpRequestContext httpRequestContext,
                                                        DynamicResourceRepository<M> dynamicResourceRepository) {
        Map<String, Set<String>> requestParams = httpRequestContext.getRequestParameters();
        requestParams.putIfAbsent("changed_fields", Set.of(""));
        requestParams.putIfAbsent("fields", Set.of(""));

        M model;
        Optional<String> optAttributes = Optional.ofNullable(requestParams.get("attributes"))
                .flatMap(attrs -> attrs.stream().findFirst());

        if (optAttributes.isPresent()) {
            String attributes = optAttributes.get();
            try {
                Map<String, JsonNode> attr = objectMapper.readValue(attributes, new TypeReference<>() { });
                var tempResource = new Resource();
                tempResource.setAttributes(attr);
                IncomingApiFields<M> updatedModelProperties = new IncomingApiFields<>(new HashMap<>(attr.size()));
                var errorParsing = new LinkedList<DefectInfo<Defect>>();
                model = dynamicResourceRepository.parseModel(tempResource, updatedModelProperties, errorParsing);

            } catch (JsonProcessingException e) {
                throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS, CommonMsg.INCORRECT_JSON, e);
            }
        } else {
            model = dynamicResourceRepository.getApiService().getApiModel().createNewInstance();
        }

        Set<String> fieldNames = requestParams.get("fields").stream()
                .flatMap(param -> Stream.of(param.split(",")))
                .collect(Collectors.toSet());

        DependsHolder dependsHolder = dynamicResourceRepository.getApiService().getApiModel().getDepends()
                .orElseThrow(() -> new CrnkResponseStatusException(
                        HttpErrorStatusEnum.ERROR__NOT_FOUND,
                        JsonapiErrorMsg.NOT_FOUND));

        Set<String> changedFieldsNames = requestParams.get("changed_fields").stream()
                .flatMap(param -> Stream.of(param.split(",")))
                .collect(Collectors.toSet());

        for (String changedField: changedFieldsNames) {
            List<String> dependsFields = dependsHolder.getDepends().getOrDefault(changedField, Collections.emptyList());
            fieldNames.addAll(dependsFields);
        }

        Function<String, List<OrderBy>> converter = dynamicResourceRepository.customSortToOrderByConverter();
        QueryParamsContext<M> queryParamsContext = new QueryParamsContext<>(
                requestParams, authenticationFacade.getUserAuthentication(), model, converter);

        dynamicResourceRepository.getApiService().getApiFieldsService().consumeExtraQueryParams(queryParamsContext);
        Map<String, Object> defaults = dynamicResourceRepository.getApiService()
                .getApiFieldsService().getDefaults(queryParamsContext, fieldNames);
        var jsonDefaults = EntryStream.of(defaults)
                .mapValues(o -> objectMapper.convertValue(o, JsonNode.class))
                .toMap();
        Resource resource = new Resource();
        resource.setId(dynamicResourceRepository.getResourceType());
        resource.setType(dynamicResourceRepository.getResourceType() + "_defaults");
        resource.setAttributes(jsonDefaults);
        var response = new Document();
        var linksInformation = new BaseCrnkLinksInformation();
        linksInformation
                .setSelf(defaultsLink(dynamicResourceRepository.getResourceType(), httpRequestContext, requestParams));
        response.setLinks(objectMapper.valueToTree(linksInformation));
        response.setData(Nullable.of(resource));

        return response;
    }

    public <M extends ModelWithId> BaseCrnkLinksInformation getBodyLinksInformation(
            QuerySpec querySpec,
            HttpRequestContext httpRequestContext,
            ApiService<M> apiService,
            boolean allowAddFields) {
        var linksInformation = new BaseCrnkLinksInformation();
        linksInformation.setSelf(sourceLink(querySpec, httpRequestContext));
        if (allowAddFields) {
            linksInformation.setAddField(addFieldsLink(querySpec.getResourceType(),
                    httpRequestContext, apiService));
        }
        return linksInformation;
    }

    public <M extends ModelWithId> BaseCrnkLinksInformation getBodyLinksInformation(
            QuerySpec querySpec,
            HttpRequestContext httpRequestContext,
            ApiService<M> apiService) {
        return getBodyLinksInformation(querySpec, httpRequestContext, apiService, true);
    }

    public String defaultsLink(String resourceType, HttpRequestContext httpRequestContext,
                               Map<String, Set<String>> requestParams) {
        return httpRequestContext.getBaseUrl() + "/" +
                resourceType + "/" +
                "defaults" +
                queryString(requestParams);
    }

    public <M extends ModelWithId> BaseCrnkMetaInformation getBodyMetaInformation(
            QuerySpec querySpec,
            FilterNode filterNode,
            long modelCount,
            ApiService<M> apiService) {
        var baseCrnkMetaInformation = new BaseCrnkMetaInformation();
        baseCrnkMetaInformation.setCount(modelCount);
        baseCrnkMetaInformation.setFields(apiService.getApiModel().getApiFieldMap().keySet());
        baseCrnkMetaInformation.setFilterDescriptions(
                filterDescriptionService.renderDescriptions(apiService
                        .getApiModel().getFilters().values()));
        if (QuerySpecUtil.isTotalRequested(querySpec)) {
            baseCrnkMetaInformation.setTotalResourceCount(apiService.count(filterNode));
        }

        return fillBodyMetaInformation(baseCrnkMetaInformation, querySpec, apiService);
    }

    public <M extends ModelWithId> BaseCrnkMetaInformation fillBodyMetaInformation(
            BaseCrnkMetaInformation baseCrnkMetaInformation,
            QuerySpec querySpec,
            ApiService<M> apiService) {
        baseCrnkMetaInformation.setFields(getIncludedFieldNames(querySpec));

        fillRightMetaInformation(baseCrnkMetaInformation, apiService);
        apiService.fillMetaInformation(baseCrnkMetaInformation);

        return baseCrnkMetaInformation;
    }

    public Set<String> getIncludedFieldNames(QuerySpec querySpec) {
        var fields = StreamEx.of(querySpec.getIncludedFields())
                .map(AbstractPathSpec::getAttributePath)
                .flatMap(Collection::stream)
                .groupingBy(Function.identity(), Collectors.counting());
        var duplicated = StreamEx.of(fields.entrySet()).filter(it -> it.getValue() > 1)
                .map(Map.Entry::getKey).collect(Collectors.joining(", "));
        if (StringUtils.isNotEmpty(duplicated)) {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS,
                    MsgWithArgs.of(
                            JsonapiErrorMsg.DUPLICATED_FIELDS,
                            duplicated)
            );
        }
        return fields.keySet();
    }

    private <M extends ModelWithId> BaseCrnkMetaInformation fillRightMetaInformation(
            BaseCrnkMetaInformation baseCrnkMetaInformation,
            ApiService<M> apiService) {
        UserAuthentication userAuthentication = authenticationFacade.getUserAuthentication();

        if (userAuthentication.userHasRight(getFullRightName("view_field__login", apiService))) {
            baseCrnkMetaInformation.setCanViewOther(true);
        }

        if (userAuthentication.userHasRight(getFullRightName("add_other", apiService))) {
            baseCrnkMetaInformation.setCanEditOther(true);
        }

        return baseCrnkMetaInformation;
    }

    private <M extends ModelWithId> String getFullRightName(String shortRightName, ApiService<M> apiService) {
        return apiService.getApiModel().getResourceType().concat("_").concat(shortRightName);
    }

    public MetaInformation fillBodyMetaInformation(BaseCrnkMetaInformation baseCrnkMetaInformation,
                                                   Collection<Resource> resources) {
        if (resources.size() == 1) {
            var editedFields = new ArrayList<>(resources).get(0).getAttributes().keySet();
            baseCrnkMetaInformation.setFields(editedFields);
        }

        return baseCrnkMetaInformation;
    }

    public LinksInformation selfLinksInformation(Resource resource,
                                                 HttpRequestContext httpRequestContext) {
        BaseCrnkLinksInformation baseCrnkLinksInformation = new BaseCrnkLinksInformation();
        baseCrnkLinksInformation.setSelf(selfLink(httpRequestContext, resource.getId()));
        return baseCrnkLinksInformation;
    }

    public <M extends ModelWithId> LinksInformation relationshipLinksInformation(Resource resource,
                                                                                 HttpRequestContext httpRequestContext,
                                                                                 ApiRelationship<M> apiRelationship) {
        BaseCrnkLinksInformation baseCrnkLinksInformation = new BaseCrnkLinksInformation();
        baseCrnkLinksInformation.setSelf(
                selfLink(httpRequestContext, resource.getId())
                        + "/" + apiRelationship.getField());
        baseCrnkLinksInformation.setRelated(selfLink(httpRequestContext, resource.getId())
                + "/relationships/" + apiRelationship.getField());

        return baseCrnkLinksInformation;
    }

    public LinksInformation getLinksForRelationship(HttpRequestContext httpRequestContext) {
        if (!httpRequestContext.getPath().contains("relationships")) {
            return null;
        }
        var linksInformation = new BaseCrnkLinksInformation();
        linksInformation.setSelf(httpRequestContext.getBaseUrl() + httpRequestContext.getPath());
        linksInformation.setRelated(httpRequestContext.getBaseUrl() + httpRequestContext.getPath()
                .replaceAll("relationships/", "")
        );

        return linksInformation;
    }

    public String selfLink(HttpRequestContext httpRequestContext, String id) {
        String path = httpRequestContext.getPath();

        if (path.endsWith("/")) {
            path = path.substring(0, path.length() - 1);
        }

        if (!path.contains(id)) {
            path += "/" + id;
        }

        return httpRequestContext.getBaseUrl() + path;
    }

    private String sourceLink(QuerySpec querySpec, HttpRequestContext httpRequestContext) {
        String path = httpRequestContext.getBaseUrl() + httpRequestContext.getPath();

        String queryString = queryString(querySpec);

        return path + queryString;
    }

    public String queryString(QuerySpec querySpec) {
        Map<String, Set<String>> params = QuerySpecUtil.getOriginalParameters(querySpec);
        return queryString(params);
    }


    private String queryString(Map<String, Set<String>> requestParams) {
        Set<Map.Entry<String, Set<String>>> requestParamsEntries = requestParams.entrySet();
        if (requestParamsEntries.isEmpty()) {
            return "";
        }

        return requestParamsEntries.stream()
                .flatMap(entry -> entry.getValue().stream().map(v -> entry.getKey() + "="
                        + URLEncoder.encode(v, StandardCharsets.UTF_8)))
                .collect(Collectors.joining("&", "?", ""));
    }

    public <M extends ModelWithId> String addFieldsLink(String resourceType, HttpRequestContext httpRequestContext,
                                                        ApiService<M> apiService) {
        if (apiService.apiCanAdd()) {
            return httpRequestContext.getBaseUrl() + "/" + resourceType + "/add_fields";
        } else {
            return null;
        }
    }

    public <M extends ModelWithId> List<ApiRelationship<M>> getApiRelationships(QuerySpec querySpec,
                                                                                ApiService<M> apiService) {
        return querySpec.getIncludedRelations().stream()
                .map(includeRelationSpec -> {
                    var apiRelationShip = apiService.getApiRelationships(includeRelationSpec.getAttributePath().get(0));
                    if (apiRelationShip == null) {
                        // по идее сюда не должны попасть, crnk уже отсекает это на своем уровне
                        throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS);
                    }
                    return apiRelationShip;
                })
                .collect(Collectors.toList());
    }

    public <M extends ModelWithId> Set<String> getApiIncludedFields(List<ApiRelationship<M>> apiRelationships) {
        return apiRelationships.stream()
                .map(ApiRelationship::getFieldsForCalcId)
                .flatMap(List::stream)
                .collect(Collectors.toSet());
    }

    /**
     * Из переданного списка запрошенных полей формируется список полей модели Crnk, которые нужно обрабатывать
     *
     * @param querySpec - строка запроса Crnk
     * @return - набор имён полей модели
     */
    public Set<String> getRequestedFieldsWithModelDefaults(QuerySpec querySpec) {
        // Дефолтное поведение - если запросили список полей возвращаем его, иначе возвращаем только id
        if (querySpec != null && !CollectionUtils.isEmpty(querySpec.getIncludedFields())) {
            return getIncludedFieldNames(querySpec);
        } else {
            return Set.of("id");
        }
    }

    public <M extends ModelWithId> Map<String, Relationship> getRelationships(String id, String url,
                                                                              ApiModel<M> apiModel) {

        if (!url.endsWith("/")) {
            url += "/";
        }
        if (!url.contains(id)) {
            url += id + "/";
        }

        String finalUrl = url;
        return apiModel.getRelationshipsLinks().entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, map -> {
                    Relationship relationship = new Relationship();
                    relationship.setLinks(objectMapper.valueToTree(
                            map.getValue().entrySet()
                                    .stream()
                                    .collect(Collectors.toMap(
                                            Map.Entry::getKey,
                                            entry -> finalUrl
                                                    + entry.getValue()))
                    ));
                    return relationship;
                }));
    }
}
