package ru.yandex.partner.jsonapi.service

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Value
import org.springframework.http.HttpStatus
import org.springframework.http.ResponseEntity
import org.springframework.stereotype.Service
import reactor.core.publisher.Mono
import ru.yandex.partner.jsonapi.controller.meta.UacSaasMetaInformation
import ru.yandex.partner.jsonapi.response.dto.UacSaasAppDto
import ru.yandex.partner.libs.extservice.saas.filter.SaasFilter
import ru.yandex.partner.libs.extservice.saas.filter.SaasFilterExpression
import ru.yandex.partner.libs.extservice.saas.response.AttributeNames
import ru.yandex.partner.libs.extservice.saas.response.Grouping
import ru.yandex.partner.libs.extservice.saas.response.SaasResponse
import ru.yandex.partner.libs.extservice.saas.service.UacSaasClient

@Service
class UacSaasService @Autowired constructor(
    private val client: UacSaasClient,
    @Value("\${saas.uac.stores}") private val stores: List<String>
) {
    private val queryFields = listOf(
        AttributeNames.TITLE.saasName, AttributeNames.BUNDLE.saasName,
        AttributeNames.URL.saasName, AttributeNames.APP_ID.saasName
    )

    fun getData(
        text: String, queryFields: List<String>?, store: String?, pageNum: Int?, limit: Int,
        plainQuery: Boolean, lang: String
    ): Pair<UacSaasMetaInformation, List<UacSaasAppDto>> {
        val nonNullQueryFields = queryFields ?: this.queryFields;

        var queryLimit = limit
        val responses = if (plainQuery) {
            listOf(
                client.performNonblockingRequest(
                    text, null,
                    calcRequestLimit(limit), AttributeNames.APP_ATTR_NAMES
                )
            )
        } else {
            queryLimit = pageNum?.let { it * limit } ?: limit
            val templates = getTemplates(nonNullQueryFields, store)
            templates.stream()
                .map {
                    client.performNonblockingRequest(
                        text, it,
                        calcRequestLimit(queryLimit), AttributeNames.APP_ATTR_NAMES
                    )
                }.toList()
        }

        val saasResponses = extractSaasResponse(responses)

        val data = getUacSaasAppDtos(saasResponses, limit, lang)
        val total = getTotal(saasResponses)

        val paginatedData = pageNum?.let {
            if ((it - 1) * limit > data.size - 1) {
                emptyList()
            } else {
                data.subList((it - 1) * limit, data.size.coerceAtMost(it * limit))
            }
        } ?: data

        return Pair(
            UacSaasMetaInformation(
                total = data.size,
                hasMore = total > queryLimit,
                count = paginatedData.size
            ), paginatedData
        )
    }

    private fun extractSaasResponse(responses: List<Mono<ResponseEntity<SaasResponse>>>): List<SaasResponse> {
        // Можно выкидывать 500 если ошибки??
        // Пока решил, что просто вернем пустоту, а ошибки запишем в лог
        return responses
            .asSequence()
            .map { it.block() }
            .filter {
                if (it == null) {
                    LOGGER.error("Saas response is null")
                    false
                } else {
                    true
                }
            }
            // cast чтобы не ругалась IDE
            .filterNotNull()
            .filter {
                when {
                    it.statusCode.is2xxSuccessful -> true
                    it.statusCode == HttpStatus.BAD_GATEWAY && it.hasBody() -> {
                        LOGGER.error("Saas response code is {}, but fallback was applied", it.statusCode)
                        // we were able to parse body as SaasResponse, so it contains some grouping/total data
                        // and we may continue
                        true
                    }
                    else -> {
                        LOGGER.error("Saas response code is {}", it.statusCode)
                        false
                    }
                }
            }
            .map { it.body }
            .filter {
                if (it == null) {
                    LOGGER.error("Saas response body is null")
                    false
                } else {
                    true
                }
            }
            // cast чтобы не ругалась IDE
            .filterNotNull()
            .toList()
    }

    private fun convertToDto(grouping: Grouping, lang: String): List<UacSaasAppDto> {
        val homeRegion = REGION_MAPPING.getOrDefault(lang, DEFAULT_REGION)

        return grouping.group
            .flatMap { it.document }
            .map { UacSaasAppDto(it.relevance, it.archiveInfo.toMap(), it.archiveInfo.url) }
            .groupBy { it.appId }
            .flatMap { chooseByRegion(it, homeRegion) }
    }

    private fun chooseByRegion(entry: Map.Entry<String, List<UacSaasAppDto>>, homeRegion: String): List<UacSaasAppDto> {
        val res = entry.value.filter { app -> homeRegion == app.region }
            .ifEmpty { entry.value.filter { app -> DEFAULT_REGION == app.region } }
            .ifEmpty { listOf(entry.value.first()) }
        return if (res.size > 1) {
            listOf(res.first())
        } else {
            res
        }
    }

    private fun getTotal(responses: List<SaasResponse>): Int {
        return responses
            .map { it.totalDocCount.last() }
            .fold(0) { t, u -> t + u }
    }

    private fun getUacSaasAppDtos(responses: List<SaasResponse>, limit: Int, lang: String): List<UacSaasAppDto> {
        val dtoIterators = responses
            .mapNotNull { it.grouping }
            .flatten()
            .map { convertToDto(it, lang).iterator() }
            .toList()

        if (dtoIterators.isEmpty()) {
            return listOf()
        }

        val dtos = mutableListOf<UacSaasAppDto>()

        val iterator = if (dtoIterators.size == 1) {
            dtoIterators[0].iterator()
        } else {
            MultiIterator(dtoIterators)
        }

        while (iterator.hasNext() && dtos.size < limit) {
            dtos.add(iterator.next())
        }

        // сортируем конечный список собранные из всех ответов
        return dtos.sortedByDescending { it.relevance }
    }

    private fun getTemplates(queryFields: List<String>, store: String?): List<String> {
        val stores = getStores(store)
        return stores.stream()
            .map {
                val orExpression = SaasFilterExpression.or(
                    queryFields.map { fields -> SaasFilter(fields, REQUEST_TEMPLATE) })
                it.let {
                    SaasFilterExpression.and(
                        SaasFilter(AttributeNames.STORE.saasName, it),
                        orExpression
                    )
                }.saasQuery
            }
            .toList()
    }

    private fun getStores(store: String?): List<String> {
        return if (store == null) {
            stores
        } else {
            listOf(store)
        }
    }

    private fun calcRequestLimit(limit: Int): Int {
        return limit * COLLAPSE_FACTOR
    }

    companion object {
        @JvmStatic
        private val LOGGER = LoggerFactory.getLogger(UacSaasService::class.java)

        private const val COLLAPSE_FACTOR = 3 // коэффициент схлопывания
        private const val REQUEST_TEMPLATE = "%request%"
        private const val DEFAULT_REGION: String = "us"
        private val REGION_MAPPING: Map<String, String> = mapOf(
            "ru" to "ru",
            "en" to "us",
            "uk" to "ua",
            "kk" to "kz",
            "be" to "by",
            "tr" to "tr"
        )
    }
}
