package ru.yandex.partner.jsonapi.utils;

import java.io.IOException;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.PathHelper;
import ru.yandex.partner.core.validation.defects.DefectInfoBuilder;
import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsService;
import ru.yandex.partner.jsonapi.crnk.fields.IncomingApiFields;
import ru.yandex.partner.jsonapi.crnk.fields.ModelPath;
import ru.yandex.partner.jsonapi.crnk.fields.ModelPathForApi;
import ru.yandex.partner.jsonapi.jackson.CollectingErrorsParser;
import ru.yandex.partner.jsonapi.messages.CommonMsg;
import ru.yandex.partner.jsonapi.models.ApiModel;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;
import ru.yandex.partner.libs.i18n.MsgWithArgs;

import static org.springframework.core.annotation.AnnotatedElementUtils.hasMetaAnnotationTypes;
import static ru.yandex.partner.jsonapi.messages.CommonMsg.UNKNOWN_FIELDS;

public class ApiUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(ApiUtils.class);

    private ApiUtils() {

    }

    public static <M extends ModelWithId> Multimap<Model, ModelProperty<?, ?>> parseJsonMap(
            M model,
            Map<String, JsonNode> updateMap,
            Map<String, ApiField<M>> apiFieldMap,
            IncomingApiFields<M> incomingApiFields,
            LinkedList<DefectInfo<Defect>> errorParsing) throws IOException {

        var collectingErrorsParser = new CollectingErrorsParser();

        var unknownFields = Sets.difference(updateMap.keySet(), apiFieldMap.keySet());
        if (!unknownFields.isEmpty()) {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS,
                    MsgWithArgs.of(UNKNOWN_FIELDS, unknownFields));
        }

        var deprecatedFields = new HashMap<String, String>();

        for (Map.Entry<String, JsonNode> entry : updateMap.entrySet()) {
            var apiField = apiFieldMap.get(entry.getKey());

            if (apiField.getDeprecated() != null) {
                deprecatedFields.put(entry.getKey(), apiField.getDeprecated());
            }

            if (!apiField.hasCoreValue()) {
                incomingApiFields.put(apiField.getJsonName(), apiField);
                continue;
            }

            try {
                var defectInfo = apiField.tryFillCoreValue(model, entry.getValue(), collectingErrorsParser);
                if (defectInfo != null) {
                    errorParsing.addAll(defectInfo);
                } else {
                    incomingApiFields.put(apiField.getJsonName(), apiField);
                }
                errorParsing.addAll(collectingErrorsParser.getAndClearCollectedDefects(apiField.getJsonName()));
            } catch (IllegalStateException e) {
                var defectInfo = DefectInfoBuilder.of(CommonMsg.INCORRECT_JSON)
                        .withPath(PathHelper.pathFromStrings(apiField.getJsonName()))
                        .<Defect>build();

                errorParsing.add(defectInfo);
            } catch (IOException e) {
                LOGGER.warn("Error while parsing", e);

                throw e;
            }
        }

        if (!deprecatedFields.isEmpty()) {
            LOGGER.warn("Found deprecated fields in current request {}", deprecatedFields);
        }

        return collectingErrorsParser.getParsedPropertiesPerModel();
    }

    @SuppressWarnings({"unchecked"})
    public static <M extends ModelWithId> ModelChanges<M> generateModelChanges(
            M modelFromDb, M modelFromApiWithChanges,
            IncomingApiFields<M> updatedFields, ApiModel<M> apiModel,
            ApiFieldsService<M> apiFieldsService) {
        ModelChanges<M> modelChanges = new ModelChanges<>(modelFromDb.getId(), apiModel.getModelClass());

        // обычные поля
        updatedFields.values().stream()
                .filter(apiField -> !apiField.getModelPath().isNested())
                .map(ApiField::getModelPath)
                .map(ModelPathForApi.class::cast)
                .forEach(mp -> modelChanges.process(mp.get(modelFromApiWithChanges), mp.rootProperty()));

        // nested поля
        updatedFields.values().stream()
                .filter(apiField -> apiField.getModelPath().isNested())
                .map(ApiField::getModelPath)
                .map(ModelPathForApi.class::cast)
                .forEach(modelPath -> {
                    // получили вложенное значение с частичным заполнением (например strategy)
                    Model nestedModelFromApi = (Model) modelPath.rootProperty().get(modelFromApiWithChanges);
                    // получили вложенное значение из модели из базы (например strategy)
                    Model nestedModelFromDb = (Model) modelPath.rootProperty().get(modelFromDb);

                    if (nestedModelFromApi != null && nestedModelFromDb != null) {
                        // Выбираем ModelProperty, которые есть в sourceModel,
                        // но которые не изменялись (не входят в fields)
                        apiFieldsService.getApiFields().stream()
                                .filter(apiField -> apiField.getModelPath() != null)
                                .filter(apiField -> modelPath.rootProperty()
                                        .equals(apiField.getModelPath().rootProperty()))
                                .filter(apiField -> !updatedFields.containsKey(apiField.getJsonName()))
                                .map(ApiField::getModelPath)
                                .map(ModelPath.class::cast)
                                .forEach(mp -> {
                                    Object o = mp.get(modelFromDb);
                                    mp.set(modelFromApiWithChanges, o);
                                });
                    }

                    modelChanges.process(nestedModelFromApi, modelPath.rootProperty());
                });

        return modelChanges;
    }

    /**
     * Получает ручки, заведенные через спринговый [Rest]Controller
     * Выбрасывает урлы с URI patterns https://docs.spring.io/spring-framework/docs/5.3.19/reference/html/web.html#mvc-ann-requestmapping-uri-templates
     * @param mapping
     * @return
     */
    public static Set<String> getSpringEndpoints(RequestMappingHandlerMapping mapping) {
        return mapping.getHandlerMethods().entrySet().stream()
                .filter(e -> hasMetaAnnotationTypes(e.getValue().getMethod(), RequestMapping.class))
                .flatMap(e -> e.getKey().getDirectPaths().stream()) // directPath don't have placeholders {...}
                .map(p -> p.endsWith("/") ? p : p.concat("/")) // add trailing slash if missing
                .collect(Collectors.toSet());
    }
}
