package ru.yandex.partner.jsonapi.validation;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.PathNodeConverter;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;

public class CovariantPathNodeConverterProvider implements PathNodeConverterProvider {
    private static final PathNodeConverter NULL_CONVERTER = new PathNodeConverter() {
        @Override
        public Path convert(PathNode.Field field) {
            return null;
        }

        @Override
        public Path convert(PathNode.Field field, PathNode.Index index) {
            return null;
        }
    };

    private final Map<Class<?>, PathNodeConverter> converterCache;
    private final Map<Class<?>, ModelPropertyPathNodeConverter> converters;
    private final PathNodeConverterProvider fallback;

    protected CovariantPathNodeConverterProvider(Builder builder) {
        this.converters = builder.converterBuilders.entrySet().stream()
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        entry -> entry.getValue().build()
                ));
        this.converterCache = new ConcurrentHashMap<>(this.converters.size());
        this.fallback = builder.fallback;
    }

    @Override
    public PathNodeConverter getConverter(Class clazz) {
        PathNodeConverter converter = converterCache.computeIfAbsent(clazz, this::getConverterInternal);
        if (converter == NULL_CONVERTER) {
            return fallback.getConverter(clazz);
        }
        return converter;
    }

    private PathNodeConverter getConverterInternal(Class<?> clazz) {
        PathNodeConverter converter = converters.get(clazz);
        if (converter != null) {
            return converter;
        }

        return converters.entrySet()
                .stream()
                .filter(entry -> entry.getKey().isAssignableFrom(clazz))
                .map(Map.Entry::getValue)
                .collect(Collectors.collectingAndThen(
                        Collectors.toList(),
                        converters -> createCompositeConverter(clazz.getSimpleName(), converters)
                ));
    }

    private PathNodeConverter createCompositeConverter(String name, List<ModelPropertyPathNodeConverter> converters) {
        if (converters.isEmpty()) {
            return NULL_CONVERTER;
        }

        return ModelPropertyPathNodeConverter.merge(name, converters);
    }

    public static class Builder {
        private PathNodeConverterProvider fallback;
        private Map<Class<?>, ModelPropertyPathNodeConverter.Builder> converterBuilders = new LinkedHashMap<>();

        public Builder converterForClass(
                Class<?> clazz,
                Consumer<ModelPropertyPathNodeConverter.Builder> pathNodeConverterConfigurer
        ) {
            pathNodeConverterConfigurer.accept(
                    converterBuilders.computeIfAbsent(clazz,
                            k -> ModelPropertyPathNodeConverter.builder(clazz.getSimpleName()))
            );

            return this;
        }

        public Builder fallback(PathNodeConverterProvider fallback) {
            this.fallback = fallback;
            return this;
        }

        public CovariantPathNodeConverterProvider build() {
            return new CovariantPathNodeConverterProvider(this);
        }
    }
}
