package ru.yandex.partner.jsonapi.validation;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathHelper;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.PathNodeConverter;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.concat;
import static ru.yandex.direct.validation.result.PathHelper.path;

public class ModelPropertyPathNodeConverter implements PathNodeConverter {
    private final String name;
    private final Map<PathNode.Field, Path> singleItemDict;
    private final Map<PathNode.Field, Path> listItemDict;

    protected ModelPropertyPathNodeConverter(Builder builder) {
        this(
                builder.name,
                EntryStream.of(builder.singleItemDict)
                        .mapKeys(PathHelper::field)
                        .mapValues(names -> mapList(names, PathHelper::field))
                        .mapValues(Path::new)
                        .toImmutableMap(),
                EntryStream.of(builder.listItemDict)
                        .mapKeys(PathHelper::field)
                        .mapValues(names -> mapList(names, PathHelper::field))
                        .mapValues(Path::new)
                        .toImmutableMap()
        );
    }

    protected ModelPropertyPathNodeConverter(
            String name,
            Map<PathNode.Field, Path> singleItemDict,
            Map<PathNode.Field, Path> listItemDict
    ) {
        this.name = name;
        this.singleItemDict = singleItemDict;
        this.listItemDict = listItemDict;
    }

    public static ModelPropertyPathNodeConverter merge(String name,
                                                       Collection<ModelPropertyPathNodeConverter> converters) {
        Map<PathNode.Field, Path> singleItems = converters.stream()
                .map(it -> it.singleItemDict)
                .flatMap(EntryStream::of)
                .collect(Collectors.toUnmodifiableMap(Map.Entry::getKey, Map.Entry::getValue));

        Map<PathNode.Field, Path> listItems = converters.stream()
                .map(it -> it.listItemDict)
                .flatMap(EntryStream::of)
                .collect(Collectors.toUnmodifiableMap(Map.Entry::getKey, Map.Entry::getValue));

        return new ModelPropertyPathNodeConverter(name, singleItems, listItems);
    }

    @Override
    public Path convert(PathNode.Field fieldName) {
        if (!singleItemDict.containsKey(fieldName)) {
            return path(fieldName);
        } else {
            return singleItemDict.get(fieldName);
        }
    }

    @Override
    public Path convert(PathNode.Field fieldName, PathNode.Index sourceIndex) {
        if (!listItemDict.containsKey(fieldName)) {
            return path(sourceIndex);
        } else {
            return concat(sourceIndex, listItemDict.get(fieldName));
        }
    }

    @Override
    public String toString() {
        return "ModelPropertyPathNodeConverter{" +
                "name='" + name + '\'' +
                '}';
    }

    public static ModelPropertyPathNodeConverter.Builder builder(String name) {
        return new Builder(name);
    }

    public static class Builder {
        private final String name;
        private final Map<String, List<String>> singleItemDict = new HashMap<>();
        private final Map<String, List<String>> listItemDict = new HashMap<>();

        public Builder(String name) {
            this.name = name;
        }

        public Builder replace(ModelProperty<?, ?> sourceProp, String destFieldName) {
            checkArgument(!singleItemDict.containsKey(sourceProp.name()), "dictionary already contains %s",
                    sourceProp.name());
            singleItemDict.put(sourceProp.name(), singletonList(destFieldName));
            return this;
        }

        public Builder replace(ModelProperty<?, ?> sourceProp, List<String> destFieldNames) {
            checkArgument(!singleItemDict.containsKey(sourceProp.name()), "dictionary already contains %s",
                    sourceProp.name());
            singleItemDict.put(sourceProp.name(), destFieldNames);
            return this;
        }

        public Builder skip(ModelProperty<?, ?> sourceProp) {
            checkArgument(!singleItemDict.containsKey(sourceProp.name()), "dictionary already contains %s",
                    sourceProp.name());
            singleItemDict.put(sourceProp.name(), emptyList());
            return this;
        }

        public Builder appendListItems(ModelProperty<?, ? extends List<?>> sourceProp, String appendingItem) {
            checkArgument(!listItemDict.containsKey(sourceProp.name()), "dictionary already contains %s",
                    sourceProp.name());
            listItemDict.put(sourceProp.name(), singletonList(appendingItem));
            return this;
        }

        public Builder appendListItems(ModelProperty<?, ? extends List<?>> sourceProp, List<String> appendingItems) {
            checkArgument(!listItemDict.containsKey(sourceProp.name()), "dictionary already contains %s",
                    sourceProp.name());
            listItemDict.put(sourceProp.name(), appendingItems);
            return this;
        }

        public ModelPropertyPathNodeConverter build() {
            return new ModelPropertyPathNodeConverter(this);
        }
    }
}
