package ru.yandex.partner.jsonapi;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Clock;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.google.common.collect.Maps;
import io.crnk.core.engine.registry.ResourceRegistry;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.InsertValuesStep2;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ArgumentsSource;
import org.mockito.Mockito;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.security.core.Authentication;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.client.WebClient;

import ru.yandex.partner.core.configuration.BkDataTestConfiguration;
import ru.yandex.partner.core.entity.block.type.tags.TagDto;
import ru.yandex.partner.core.entity.block.type.tags.TagService;
import ru.yandex.partner.core.entity.utils.ConditionUtils;
import ru.yandex.partner.core.junit.MySqlRefresher;
import ru.yandex.partner.dbschema.partner.tables.UserFeatures;
import ru.yandex.partner.dbschema.partner.tables.Users;
import ru.yandex.partner.dbschema.partner.tables.records.UserFeaturesRecord;
import ru.yandex.partner.defaultconfiguration.PartnerLocalDateTime;
import ru.yandex.partner.jsonapi.utils.UrlUtils;
import ru.yandex.partner.libs.auth.annotation.AuthenticationType;
import ru.yandex.partner.libs.auth.model.AuthenticationMethod;
import ru.yandex.partner.libs.auth.model.TvmUserAuthentication;
import ru.yandex.partner.libs.auth.model.UserAuthenticationHolder;
import ru.yandex.partner.libs.auth.model.UserCredentials;
import ru.yandex.partner.libs.auth.provider.tvm.TvmAuthenticationProvider;
import ru.yandex.partner.libs.extservice.balance.BalanceService;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.BalancePartnerContract;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Bank;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Collateral;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Contract;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Person;
import ru.yandex.partner.libs.rbac.userrole.Pair;
import ru.yandex.partner.test.http.json.model.TestHttpCase;
import ru.yandex.partner.test.http.json.model.TestHttpExchange;
import ru.yandex.partner.test.http.json.model.TestHttpResponse;
import ru.yandex.partner.test.http.json.utils.TestCaseManager;
import ru.yandex.partner.test.utils.TestUtils;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyLong;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.reset;

/**
 * https://relentlesscoding.com/posts/automatic-rollback-of-transactions-in-spring-tests/
 * ${@link org.springframework.transaction.annotation.Transactional}
 * Transactions are not rolled back when the code that is invoked does not interact with the database. An example
 * would be writing an end-to-end test that uses RestTemplate to make an HTTP request to some endpoint that then
 * makes a modification in the database. Since the RestTemplate does not interact with the database (it just creates
 * and sends an HTTP request), @Transactional will not rollback anything that is an effect of that HTTP request. This
 * is because a separate transaction, one not controlled by the test, will be started.
 */
@JsonApiTest
@ExtendWith(MySqlRefresher.class)
public class ApiTest {
    private static final Logger LOGGER = LoggerFactory.getLogger(ApiTest.class);

    private final DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

    private static boolean needGitUpdate;
    private static boolean needSelfUpdate;

    private String host;

    @Nullable
    private Map<String, List<String>> fields;

    @LocalServerPort
    private int port;

    @Autowired
    private ResourceRegistry resourceRegistry;

    @Autowired
    private DSLContext dslContext;

    @Autowired
    private BalanceService balanceService;

    @Autowired
    private TagService tagService;

    @Autowired
    private TvmAuthenticationProvider tvmAuthenticationProvider;

    @Autowired
    private BkDataTestConfiguration.BlockWithCustomBkDataValidatorProviderStub blockBkDataValidatorProvider;

    @BeforeAll
    static void beforeAll() {
        if (TestCaseManager.needGitDownload()) {
            TestCaseManager.gitDownload(
                    JsonFileArgumentProvider.getDefaultSourcePath(),
                    JsonFileArgumentProvider.getGitDirPath()
            );
            Assertions.fail("File(s) downloaded from git");
        }

        needGitUpdate = TestCaseManager.needGitUpdate();
        needSelfUpdate = TestCaseManager.needSelfUpdate();
    }

    @BeforeEach
    void setUp() {
        host = "http://127.0.0.1:" + port;
    }

    @ParameterizedTest
    @ArgumentsSource(JsonFileArgumentProvider.class)
    void requestResponseTest(String resourcePath) throws IOException {
        if (needGitUpdate) {
            TestCaseManager.gitUpdate(resourcePath);
            Assertions.fail("Resource updated from git. Path = " + resourcePath);
        }
        testByResource(resourcePath);

    }

    private void testByResource(String resourcePath) throws IOException {
        TestHttpCase testHttpCase = TestHttpCase.load(resourcePath);
        resetMocks(testHttpCase);
        var tests = StreamEx.of(testHttpCase.getTests()).filter(it -> !Boolean.TRUE.equals(it.getSkip())).toList();

        testByTestHttpExchanges(tests, needSelfUpdate);
        if (needSelfUpdate) {
            TestCaseManager.selfUpdate(resourcePath, testHttpCase);
            Assertions.fail("Resource self_updated. Path = " + resourcePath);
        }
    }

    private void testByTestHttpExchanges(List<TestHttpExchange> testHttpExchanges, boolean selfUpdate) {
        TestHttpResponse lastCreated = null;
        TestHttpResponse lastChanged = null;
        for (TestHttpExchange exchange : testHttpExchanges) {
            substituteAllFieldsVariable(exchange, lastChanged, lastCreated);
            var addedFeatures = mockFeaturePerTest(exchange.getRequest().getMockedFeatures());

            var actualResponse = new TestHttpResponse(exchange.getRequest().perform(host));
            var expectedResponse = exchange.getResponse();

            if (selfUpdate) {
                exchange.setResponse(actualResponse);
            } else {
                expectedResponse.assertEquals(actualResponse, exchange.getRequest().getName());
            }
            removeMockedFeature(addedFeatures);
            lastChanged = getLastChanged(exchange, lastChanged, actualResponse);
            lastCreated = getLastCreated(exchange, lastCreated, actualResponse);
        }
    }

    private void substituteAllFieldsVariable(TestHttpExchange exchange,
                                             TestHttpResponse lastChanged,
                                             TestHttpResponse lastCreated) {
        exchange.getRequest().setUrl(UrlUtils.replaceLastChangeId(lastChanged, exchange.getRequest().getUrl()));
        exchange.getRequest().setBody(UrlUtils.replaceLastChangeId(lastChanged, exchange.getRequest().getBody()));
        exchange.getRequest().setUrl(UrlUtils.replaceLastCreatedId(lastCreated, exchange.getRequest().getUrl()));
        exchange.getRequest().setBody(UrlUtils.replaceLastCreatedId(lastCreated, exchange.getRequest().getBody()));
        exchange.getRequest().setUrl(UrlUtils.replaceAllFields(resourceRegistry, exchange.getRequest().getUrl()));

        if (fields != null) {
            exchange.getRequest().setUrl(UrlUtils.replaceFields(fields, exchange.getRequest().getUrl()));
        }
    }

    private Map<Long, Set<String>> mockFeaturePerTest(JsonNode mockNodeFeatures) {
        if (mockNodeFeatures != null) {
            var mockFeaturePerTest = TestUtils.parseNode(mockNodeFeatures, "mock_features", MockFeatures.class);
            if (!mockFeaturePerTest.isEmpty()) {
                return mockFeatures(mockFeaturePerTest);
            }
        }
        return Collections.emptyMap();
    }

    private TestHttpResponse getLastCreated(TestHttpExchange curExchange, TestHttpResponse lastCreated,
                                            TestHttpResponse actualResponse) {
        if (actualResponse.getStatus() > 299) {
            return lastCreated;
        }
        if ("post".equals(curExchange.getRequest().getMethod())
                && !curExchange.getRequest().getUrl().contains("/action")) {
            return actualResponse;
        } else {
            return lastCreated;
        }
    }

    private TestHttpResponse getLastChanged(TestHttpExchange curExchange, TestHttpResponse lastChanged,
                                            TestHttpResponse actualResponse) {
        if (actualResponse.getStatus() > 299) {
            return lastChanged;
        }

        if ("patch".equals(curExchange.getRequest().getMethod())) {
            return actualResponse;
        } else if ("post".equals(curExchange.getRequest().getMethod())
                && curExchange.getRequest().getUrl().contains("/action/edit")) {
            return actualResponse;
        } else if ("post".equals(curExchange.getRequest().getMethod())
                // todo в перл считается как changed не created
                && curExchange.getRequest().getUrl().contains("/action/duplicate")) {
            return actualResponse;
        } else {
            return lastChanged;
        }
    }

    private void resetMocks(TestHttpCase testHttpCase) {
        var mockBalance = testHttpCase.getOption("mock_balance", Boolean.class);
        mockBalance(Boolean.TRUE.equals(mockBalance));

        var mockYtPartner = testHttpCase.getOption("mock_yt_partner", Boolean.class);
        mockYtPartner(Boolean.TRUE.equals(mockYtPartner));

        var mockFeatures = testHttpCase.getOption("mock_features", MockFeatures.class);

        mockFeatures(mockFeatures);

        var currentTime = testHttpCase.getOption("mock_curdate", String.class);
        mockDateTime(currentTime);

        Long mockedTvmUser = testHttpCase.getOption("mock_tvm", Long.class);
        mockTvm(mockedTvmUser);

        var mockBkDataValidation = testHttpCase.getOption("mock_bk_data_validator", Boolean.class);
        mockBkDataValidator(Boolean.TRUE.equals(mockBkDataValidation));

        var tf = TypeFactory.defaultInstance();
        fields = testHttpCase.getOption("fields",
                tf.constructMapLikeType(
                        Map.class,
                        tf.constructType(String.class),
                        tf.constructCollectionLikeType(List.class, String.class)
                )
        );
    }

    private void mockTvm(Long mockedTvmUser) {
        if (mockedTvmUser != null) {
            doReturn(new UserAuthenticationHolder(new TvmUserAuthentication(
                    true,
                    AuthenticationMethod.AUTH_VIA_TVM_SERVICE,
                    new UserCredentials(mockedTvmUser)
            )))
                    .when(tvmAuthenticationProvider)
                    .authenticate(any(Authentication.class));

            doReturn(true)
                    .when(tvmAuthenticationProvider)
                    .supports(any());

            doReturn(AuthenticationType.TVM)
                    .when(tvmAuthenticationProvider)
                    .supportedAuthType();
        } else {
            Mockito.reset(tvmAuthenticationProvider);
        }
    }

    private void mockBalance(boolean mockBalance) {
        if (mockBalance) {
            // примитивный мок
            doReturn(List.of(
                    new BalancePartnerContract(
                            new Person.Builder()
                                    .withClientId(11009L)
                                    .withBik("12345678")
                                    .build(),
                            new Contract.Builder()
                                    .withDt(LocalDate.of(2019, 5, 1))
                                    .withIsSigned(LocalDate.of(2019, 5, 2))
                                    .withType("PARTNERS")
                                    .build(),
                            List.of(new Collateral.Builder()
                                    .withCollateralTypeId(2000)
                                    .withDt(LocalDate.of(2020, 6, 1))
                                    .build())
                    )
            )).when(balanceService).getPartnerContracts(anyLong());

            doReturn(new Bank(true, "12345678", "bik", "bank_name"))
                    .when(balanceService).getBankByBik(anyString());
        } else {
            reset(balanceService);
        }
    }

    private void mockYtPartner(boolean mockYtPartner) {
        if (mockYtPartner) {
            Mockito.doReturn(Set.of(1L, 2L, 3L, 4L, 5L, 6L)).when(tagService).getTagIds();
            Mockito.doReturn(new TreeMap<>(Map.of(
                    1L, new TagDto(1L, "name1", "descr1"),
                    2L, new TagDto(2L, "name2", "descr2"),
                    3L, new TagDto(3L, "name3", "descr3")
            ))).when(tagService).getTags();
        } else {
            Mockito.reset(tagService);
        }
    }

    private void mockBkDataValidator(boolean doMock) {
        blockBkDataValidatorProvider.setDisableBkDataValidation(doMock);
    }

    private void mockDateTime(@Nullable String localDateTime) {
        if (localDateTime != null) {
            try {
                PartnerLocalDateTime.setClock(Clock.fixed(
                        dateFormat.parse(localDateTime).toInstant(),
                        ZoneId.systemDefault()));
            } catch (ParseException e) {
                throw new IllegalStateException(e.getMessage());
            }
        } else {
            PartnerLocalDateTime.setClock(Clock.systemDefaultZone());
        }
    }

    private Map<Long, Set<String>> mockFeatures(@Nullable MockFeatures mockFeatures) {
        if (mockFeatures == null || mockFeatures.isEmpty()) {
            return null;
        }
        var addedFeatures = Maps.<Long, Set<String>>newHashMapWithExpectedSize(mockFeatures.size());
        var userIds = dslContext.select(Users.USERS.LOGIN, Users.USERS.ID)
                .from(Users.USERS)
                .where(Users.USERS.LOGIN.in(mockFeatures.keySet()))
                .fetchMap(Users.USERS.LOGIN, Users.USERS.ID);

        var featuresMap = dslContext.select(UserFeatures.USER_FEATURES.USER_ID, UserFeatures.USER_FEATURES.FEATURE)
                .from(UserFeatures.USER_FEATURES)
                .where(UserFeatures.USER_FEATURES.USER_ID.in(userIds.values()))
                .fetchGroups(UserFeatures.USER_FEATURES.USER_ID, UserFeatures.USER_FEATURES.FEATURE);

        var records = new ArrayList<UserFeaturesRecord>();
        for (Map.Entry<String, List<String>> entry : mockFeatures.entrySet()) {
            var userId = userIds.get(entry.getKey());
            if (userId == null) {
                LOGGER.warn("User not found. Login: {}", entry.getKey());
                continue;
            }

            var features = featuresMap.getOrDefault(userId, List.of());
            for (String feature : entry.getValue()) {
                if (!features.contains(feature)) {
                    addedFeatures.computeIfAbsent(userId, s -> new HashSet<>()).add(feature);
                    records.add(new UserFeaturesRecord(null, userId, feature));
                }
            }
        }

        if (!records.isEmpty()) {
            var insertSetStep = dslContext.insertInto(UserFeatures.USER_FEATURES,
                    UserFeatures.USER_FEATURES.USER_ID, UserFeatures.USER_FEATURES.FEATURE);

            // TODO в jooq 3.15 появился метод valuesOfRecords
            //  https://www.jooq.org/doc/3.15/manual-single-page/#insert-valuesco
            InsertValuesStep2<UserFeaturesRecord, Long, String> insertValuesStep = null;
            for (UserFeaturesRecord record : records) {
                insertValuesStep = insertSetStep.values(record.getUserId(), record.getFeature());
            }

            insertValuesStep.execute();
        }
        return addedFeatures;
    }

    private void removeMockedFeature(@Nonnull Map<Long, Set<String>> addedFeatures) {
        if (addedFeatures.isEmpty()) {
            return;
        }
        dslContext.delete(UserFeatures.USER_FEATURES).where(ConditionUtils.toMapCondition(addedFeatures,
                UserFeatures.USER_FEATURES.USER_ID, UserFeatures.USER_FEATURES.FEATURE)).execute();

    }

    private void testWebClientWithHeaders(
            final List<Pair<String, String>> headersToSet,
            final HttpStatus expectedStatus,
            final String expectedMessage
    ) {
        try {
            final WebClient.RequestBodySpec webClient = WebClient.create()
                    .method(HttpMethod.PATCH)
                    .uri(new URI(host + "/restapi/v1/users/1009"));

            webClient.body(BodyInserters.fromValue("{\"data\":{\"attributes\":{\"midname\":\"mid_name\"}," +
                    "\"id\":\"1009\",\"type\":\"users\"}}"));

            headersToSet.forEach(it -> webClient.header(it.getFirst(), it.getSecond()));

            var response = webClient.exchangeToMono(clientResponse -> clientResponse.toEntity(String.class))
                    .block();

            Assertions.assertNotNull(response);
            Assertions.assertEquals(expectedStatus, response.getStatusCode());
            Assertions.assertEquals(expectedMessage, response.getBody());
        } catch (URISyntaxException e) {
            Assertions.fail(e.getMessage());
        }
    }

    @Test
    void patchRequestWithoutAcceptsHeader() {
        testWebClientWithHeaders(
                List.of(
                        new Pair<>("Authorization", "token mocked-yan-partner-assistant"),
                        new Pair<>("Content-type", "application/vnd.api+json")
                ),
                HttpStatus.NOT_ACCEPTABLE,
                "{\"errors\":[{\"id\":\"8\",\"code\":\"406\",\"title\":\"Not Acceptable\"}]}\n"
        );
    }

    @Test
    void patchRequestWithoutContentTypeHeader() {
        testWebClientWithHeaders(
                List.of(
                        new Pair<>("Authorization", "token mocked-yan-partner-assistant"),
                        new Pair<>("Accept", "application/vnd.api+json")
                ),
                HttpStatus.UNSUPPORTED_MEDIA_TYPE,
                "{\"errors\":[{\"id\":\"9\",\"code\":\"415\",\"title\":\"Unsupported Media Type\"}]}\n"
        );
    }

    @Test
    void postRequestActionsWithoutAccept() {
        try {
            final WebClient.RequestBodySpec webClient = WebClient.create()
                    .method(HttpMethod.POST)
                    .uri(new URI(host + "/v1/users/286573/action/edit"));

            webClient.header("Authorization", "token mocked-yan-partner-assistant");
            webClient.header("Content-Type", "application/vnd.api+json");
            webClient.body(BodyInserters.fromValue("--data '{ \"lastname\": \"lastname\n }"));

            var response = webClient.exchangeToMono(clientResponse -> clientResponse.toEntity(String.class))
                    .block();

            Assertions.assertNotNull(response);
            Assertions.assertEquals(response.getStatusCode(), HttpStatus.NOT_ACCEPTABLE);
            Assertions.assertEquals("{\"errors\":[{\"id\":\"8\",\"code\":\"406\",\"title\":\"Not " +
                    "Acceptable\"}]}\n", response.getBody());
        } catch (URISyntaxException e) {
            Assertions.fail("malformed url\n" + e.getMessage());
        }
    }

    @Test
    void protectedRequestsWithoutAcceptAndContentType() {
        try {
            final WebClient.RequestBodySpec webClient = WebClient.create()
                    .method(HttpMethod.GET)
                    .uri(new URI(host + "/root"));
            webClient.header("Authorization", "token mocked-yan-partner-assistant");

            var response = webClient.exchangeToMono(clientResponse -> clientResponse.toEntity(String.class))
                    .block();

            Assertions.assertNotNull(response);
            Assertions.assertEquals(response.getStatusCode(), HttpStatus.FOUND);
        } catch (URISyntaxException e) {
            Assertions.fail("malformed url\n" + e.getMessage());
        }
    }

    @Test
    void testPartnerDateTime() {
        PartnerLocalDateTime.setClock(Clock.systemDefaultZone());
        Assertions.assertEquals(
                LocalDateTime.now().truncatedTo(ChronoUnit.DAYS),
                PartnerLocalDateTime.now().truncatedTo(ChronoUnit.DAYS)
        );
    }

    @Test
    void getRequestWithCyrillicInCookie() {
        testWebClientWithHeaders(
                List.of(
                        new Pair<>("Authorization", "token mocked-yan-partner-assistant"),
                        new Pair<>("Accept", "application/vnd.api+json"),
                        new Pair<>("Content-type", "application/vnd.api+json"),
                        new Pair<>("Cookie", "aÑ\u008BÐ²Ñ\u008BÐ²")
                ),
                HttpStatus.BAD_REQUEST,
                "{\"errors\":[{\"id\":\"1\",\"code\":\"400\",\"title\":\"Bad Request\",\"detail\":\"The request was " +
                        "rejected because the header value \\\"aÑ\u008BÐ²Ñ\u008BÐ²\\\" is not allowed.\"}]}\n"
        );
    }

    private static class MockFeatures extends HashMap<String, List<String>> {
        // just for .class
    }
}
