package ru.yandex.partner.jsonapi;

import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.junit.jupiter.api.extension.ExtensionContext;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.ArgumentsProvider;

public class FileArgumentProvider implements ArgumentsProvider {
    private final String relativeTestResourcePath;
    private final String resourceDirectoryPath;
    private final Path sourceDir;

    public FileArgumentProvider(String relativeTestResourcePath, String envName)
            throws IOException, URISyntaxException {
        this.relativeTestResourcePath = relativeTestResourcePath;
        this.resourceDirectoryPath = getResourceDirectoryPath(envName);
        this.sourceDir = Path.of(ApiTest.class.getResource(resourceDirectoryPath).toURI());
    }

    @Override
    public Stream<? extends Arguments> provideArguments(ExtensionContext context) throws IOException {
        List<String> paths = fillPaths(sourceDir);

        return paths.stream().map(Arguments::of);
    }

    private List<String> fillPaths(Path path) throws IOException {
        return Files.walk(path)
                .filter(Files::isRegularFile)
                .map(Path::toFile)
                .filter(file -> "json".equalsIgnoreCase(FilenameUtils.getExtension(file.getName())))
                .map(file -> Paths.get(
                        resourceDirectoryPath, sourceDir.toUri().relativize(file.toURI()).getPath()).toString())
                .sorted()
                .collect(Collectors.toList());
    }

    private String getResourceDirectoryPath(String envName) throws IOException {
        String envVariableTestResourcesPath = StringUtils.isEmpty(envName) ? "" : System.getenv(envName);

        String path = "/jsonapi/src/test/resources" + relativeTestResourcePath;
        if (StringUtils.isBlank(envVariableTestResourcesPath)) {
            return relativeTestResourcePath;
        } else if (envVariableTestResourcesPath.contains(path)) {
            return envVariableTestResourcesPath.substring(
                    envVariableTestResourcesPath.lastIndexOf(relativeTestResourcePath));
        } else {
            throw new IOException("Environment variable is incorrect. It has to contain path: " +
                    path);
        }
    }
}
