package ru.yandex.partner.jsonapi.configuration;

import java.util.Map;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.jooq.DSLContext;
import org.springframework.web.reactive.function.client.WebClient;

import ru.yandex.partner.libs.auth.exception.authentication.ApiKeyAuthenticationI18nException;
import ru.yandex.partner.libs.auth.message.YandexCabinetMsg;
import ru.yandex.partner.libs.auth.model.AuthenticationMethod;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.auth.model.UserCredentials;
import ru.yandex.partner.libs.auth.provider.apikey.ApiKeyAuthenticationProvider;
import ru.yandex.partner.libs.auth.provider.apikey.ApiKeyRpcConfig;
import ru.yandex.partner.test.db.QueryLogService;

import static org.mockito.Mockito.mock;
import static ru.yandex.partner.dbschema.partner.tables.Users.USERS;

public class TestApiKeyAuthenticationProvider extends ApiKeyAuthenticationProvider {

    private static final Map<String, Long> MOCKED_USER_ID_BY_LOGIN = Map.of("passport-only-user", Long.MAX_VALUE);

    private final DSLContext dslContext;
    private final QueryLogService queryLogService;

    public TestApiKeyAuthenticationProvider(
            String apiKeyServiceToken,
            DSLContext dslContext, QueryLogService queryLogService,
            ObjectMapper objectMapper,
            ApiKeyRpcConfig apiKeyRpcConfig
    ) {
        super(
                apiKeyServiceToken,
                mock(WebClient.class),
                objectMapper,
                apiKeyRpcConfig
        );
        this.dslContext = dslContext;
        this.queryLogService = queryLogService;
    }

    @Override
    protected UserAuthentication authenticateKey(String remoteAddr, String key) {
        Long id = getUserId(key);
        if (id != null) {
            return new UserAuthentication(AuthenticationMethod.AUTH_VIA_API_KEYS_OAUTH, new UserCredentials(id));
        } else {
            throw new ApiKeyAuthenticationI18nException(YandexCabinetMsg.INVALID_KEY);
        }
    }

    private Long getUserId(String login) {
        Long value = queryLogService.executeNoLogQuery(() -> dslContext.select(USERS.ID)
                .from(USERS)
                .where(USERS.LOGIN.eq(login))
                .fetchOne(USERS.ID));
        if (value != null) {
            return value;
        } else {
            return MOCKED_USER_ID_BY_LOGIN.get(login);
        }
    }
}
