package ru.yandex.partner.jsonapi.crnk.filter.parser;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.extension.ExtensionContext;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.ArgumentsProvider;
import org.junit.jupiter.params.provider.ArgumentsSource;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.partner.core.entity.user.filter.UserFilters;
import ru.yandex.partner.core.entity.user.filter.UserModelFilterContainer;
import ru.yandex.partner.core.entity.user.model.BaseUser;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.entity.user.multistate.CanChangeRequisitesCheck;
import ru.yandex.partner.core.entity.user.multistate.UserActionChecksService;
import ru.yandex.partner.core.entity.user.multistate.UserMultistateExpressionParser;
import ru.yandex.partner.core.entity.user.multistate.UserMultistateGraph;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.core.multistate.user.UserStateFlag;
import ru.yandex.partner.jsonapi.JsonApiTest;
import ru.yandex.partner.jsonapi.crnk.CrnkModelFilters;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilter;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilterName;
import ru.yandex.partner.jsonapi.crnk.filter.parser.values.MultistateCrnkFilterValueParser;

import static org.mockito.Mockito.mock;
import static ru.yandex.partner.dbschema.partner.tables.UserRole.USER_ROLE;
import static ru.yandex.partner.dbschema.partner.tables.Users.USERS;


@JsonApiTest
class CrnkFilterParserTest {
    @Autowired
    private UserModelFilterContainer userModelFilterContainer;

    @ParameterizedTest(name = "{index} {0}")
    @ArgumentsSource(ConditionArgumentProvider.class)
    void toCondition(String filterJson, Condition expectedCondition) {
        UserMultistateGraph userMultistateGraph =
                new UserMultistateGraph(new UserActionChecksService(mock(CanChangeRequisitesCheck.class)));

        UserMultistateExpressionParser userMultistateExpressionParser = new UserMultistateExpressionParser();
        MultistateCrnkFilterValueParser<User, UserStateFlag> multistateCrnkFilterValueParser
                = new MultistateCrnkFilterValueParser<>(userMultistateGraph, userMultistateExpressionParser);

        TestUserCrnkModelFilters testUserCrnkModelFilters =
                new TestUserCrnkModelFilters(multistateCrnkFilterValueParser);

        CrnkFilterParser crnkFilterParser = new CrnkFilterParser(new ObjectMapper());

        CoreFilterNode<User> coreFilterNode = crnkFilterParser.parse(filterJson)
                .toCoreFilterNode(testUserCrnkModelFilters);
        Assertions.assertEquals(expectedCondition,
                coreFilterNode.toCondition(User.class, userModelFilterContainer));
    }

    @ParameterizedTest(name = "{index} {0}")
    @ArgumentsSource(FilterNodeArgumentProvider.class)
    void toFilterNode(String filterJson, FilterNode expectedFilterNode) {
        CrnkFilterParser crnkFilterParser = new CrnkFilterParser(new ObjectMapper());

        FilterNode actualFilterNode = crnkFilterParser.parse(filterJson);
        Assertions.assertEquals(expectedFilterNode, actualFilterNode);
    }

    private static class ConditionArgumentProvider implements ArgumentsProvider {
        @Override
        public Stream<? extends Arguments> provideArguments(ExtensionContext context) {
            return Stream.of(
                    Arguments.of("[\"login\", \"LIKE\", \"test-login\"]", USERS.LOGIN.like("%test-login%")),
                    Arguments.of("[\"login\", \"LIKE\", \"test-login$\"]", USERS.LOGIN.like("%test-login")),
                    Arguments.of("[\"login\", \"LIKE\", \"test-login\\\\$\"]", USERS.LOGIN.like("%test-login\\$%")),
                    Arguments.of("[\"login\", \"LIKE\", \"^test_login\"]", USERS.LOGIN.like("test\\_login%")),
                    Arguments.of("{\"login\": \"test-login\"}", USERS.LOGIN.eq("test-login")),
                    Arguments.of("{\"is_adfox_partner\": true}", USERS.IS_ADFOX_PARTNER.eq(1L)),
                    Arguments.of("[\"id\", \">\", \"10\"]", USERS.ID.gt(10L)),
                    Arguments.of("[\"id\", \"=>\", \"10\"]", USERS.ID.ge(10L)),
                    Arguments.of("[\"id\", \"<\", \"10\"]", USERS.ID.lt(10L)),
                    Arguments.of("[\"id\", \"=<\", \"10\"]", USERS.ID.le(10L)),
                    Arguments.of(
                            "[\"AND\",[{\"client_id\":[\"=\",\"1\"]},{\"role_id\":[\"IN\",[\"9\"]]}," +
                                    "{\"login\":[\"=\",\"test-login\"]},{\"accountant_email\":[\"=\"," +
                                    "[\"accountant_email@test.ru\"]]}," +
                                    "{\"id\":[\">\",[\"10\"]]}]]",
                            USERS.CLIENT_ID.eq(1L)
                                    .and(USERS.ID.eq(DSL.any(DSL.select(USER_ROLE.USER_ID)
                                            .from(USER_ROLE)
                                            .where(USER_ROLE.ROLE_ID.eq(9L)))))
                                    .and(USERS.LOGIN.eq("test-login"))
                                    .and(USERS.ACCOUNTANT_EMAIL.eq("accountant_email@test.ru"))
                                    .and(USERS.ID.gt(10L))
                    ),

                    Arguments.of("[\"OR\",[[\"AND\",[{\"login\": [\"=\",\"test-login\"]},[\"client_id\",\">\"," +
                                    "1000]]]," +
                                    "[{\"login\": \"like-god\"}]]]",
                            USERS.LOGIN.eq("test-login")
                                    .and(USERS.CLIENT_ID.gt(1000L))
                                    .or(USERS.LOGIN.eq("like-god"))
                    ),
                    Arguments.of("[\"multistate\", \"=\", \"__EMPTY__\"]", USERS.MULTISTATE.eq(0L)),
                    Arguments.of("[\"multistate\", \"=\", \"__EMPTY__ OR contacts_provided\"]", USERS.MULTISTATE.in(
                            new HashSet<>(List.of(0, 1, 17, 3, 19, 9, 25, 11, 27))
                    )),
                    Arguments.of("[\"multistate\", \"=\", 0]", USERS.MULTISTATE.eq(0L)),
                    Arguments.of("[\"multistate\", \"=\", [0, 1]]]", USERS.MULTISTATE.in(
                            new HashSet<>(List.of(1, 0))
                    )),
                    Arguments.of("[\"multistate\", \"=\", [0,0]]", USERS.MULTISTATE.eq(0L)),
                    Arguments.of("[\"multistate\", \"=\", [\"contacts_provided\"]]",
                            USERS.MULTISTATE.in(new HashSet<>(List.of(1, 17, 3, 19, 9, 25, 11, 27)))
                    )
            );
        }
    }

    private static class FilterNodeArgumentProvider implements ArgumentsProvider {

        @Override
        public Stream<? extends Arguments> provideArguments(ExtensionContext context) {
            FilterNode filterNode =
                    new FilterNode(new RawFilter("campaign", FilterOperator.MATCH,
                            new ObjectMapper().createArrayNode().add("all_domain").add("LIKE").add("test-domain")));

            return Stream.of(
                    Arguments.of("[\"campaign.all_domain\", \"LIKE\", \"test-domain\"]", filterNode),
                    Arguments.of("[\"campaign\", \"MATCH\", [\"all_domain\", \"LIKE\", \"test-domain\"]]", filterNode)
            );
        }
    }

    public static class TestUserCrnkModelFilters implements CrnkModelFilters<BaseUser> {
        private final Map<String, CrnkFilter<BaseUser, ?>> filters = new HashMap<>();

        private TestUserCrnkModelFilters(
                MultistateCrnkFilterValueParser<? extends BaseUser, UserStateFlag> multistateCrnkFilterValueParser
        ) {
            List<CrnkFilter<? super User, ?>> filtersList = List.of(
                    CrnkFilter.builder(FilterNameEnum.ID).
                            toCrnkFilter(UserFilters.ID),
                    CrnkFilter.builder(FilterNameEnum.CLIENT_ID).
                            toCrnkFilter(UserFilters.CLIENT_ID),
                    CrnkFilter.builder(FilterNameEnum.LOGIN).
                            toCrnkFilter(UserFilters.LOGIN),
                    CrnkFilter.builder(FilterNameEnum.ACCOUNTANT_EMAIL).
                            toCrnkFilter(UserFilters.ACCOUNTANT_EMAIL),
                    CrnkFilter.builder(FilterNameEnum.ROLE_ID).
                            toCrnkFilter(UserFilters.ROLE_ID),
                    CrnkFilter.builder(FilterNameEnum.IS_ADFOX_PARTNER).
                            toCrnkFilter(UserFilters.IS_ADFOX_PARTNER),
                    CrnkFilter.builder(FilterNameEnum.MULTISTATE).
                            toCrnkFilter(
//                                    UserFilters.MULTISTATE,
                                    UserFilters.MULTISTATE.asLongFilter(),
                                    multistateCrnkFilterValueParser
                            )
            );

            filtersList.forEach(filter -> filters.put(filter.getName(), (CrnkFilter<BaseUser, ?>) filter));
        }

        public enum FilterNameEnum implements CrnkFilterName {
            ID("id"),
            CLIENT_ID("client_id"),
            LOGIN("login"),
            ACCOUNTANT_EMAIL("accountant_email"),
            MULTISTATE("multistate"),
            ROLE_ID("role_id"),
            IS_ADFOX_PARTNER("is_adfox_partner");

            private final String name;

            FilterNameEnum(String name) {
                this.name = name;
            }

            @Override
            public String getName() {
                return name;
            }
        }

        @Override
        public Map<String, CrnkFilter<BaseUser, ?>> getFilters() {
            return filters;
        }
    }
}
