package ru.yandex.partner.jsonapi.fault.http

import org.assertj.core.api.Assertions.assertThat
import org.junit.jupiter.api.Disabled
import org.junit.jupiter.api.Test
import org.mockito.ArgumentMatchers
import org.mockito.Mockito
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Value
import org.springframework.boot.web.server.LocalServerPort
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Import
import org.springframework.http.ResponseEntity
import org.springframework.security.core.Authentication
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.RestController
import org.springframework.web.reactive.function.client.WebClient
import reactor.core.publisher.Flux
import reactor.core.publisher.Mono
import reactor.core.publisher.toFlux
import ru.yandex.partner.core.entity.QueryOpts
import ru.yandex.partner.core.entity.user.model.User
import ru.yandex.partner.core.entity.user.service.UserService
import ru.yandex.partner.jsonapi.JsonApiTest
import ru.yandex.partner.libs.annotation.PartnerTransactional
import ru.yandex.partner.libs.auth.annotation.Auth
import ru.yandex.partner.libs.auth.annotation.AuthenticationType
import ru.yandex.partner.libs.auth.model.AuthenticationMethod
import ru.yandex.partner.libs.auth.model.TvmUserAuthentication
import ru.yandex.partner.libs.auth.model.UserAuthenticationHolder
import ru.yandex.partner.libs.auth.model.UserCredentials
import ru.yandex.partner.libs.auth.provider.tvm.TvmAuthenticationProvider
import java.time.Duration
import java.util.concurrent.TimeoutException

@Disabled("This test is a Demo, it dirties context and may fail other tests")
@Import(HttpLongRequestsServiceFailureTest.Config::class)
@JsonApiTest
class HttpLongRequestsServiceFailureTest {
    @LocalServerPort
    private val port = 0

    @Autowired
    private lateinit var tvmAuthenticationProvider: TvmAuthenticationProvider

    @Value("\${mysql.max-pool-size}")
    private lateinit var sqlPoolSize: Integer

    /**
     * Тест демонстрирует обрывы соединений клиентами для запросов,
     * которые лочат данные и долго обрабатываются.
     *
     * Когда не починено (большой таймаут на локах):
     * Должны вызывать полную выборку пула запросов/соединений к БД
     * и SQL-таймауты на получении коннекшена из пула
     * вызывая "недоступность" БД для других запросов
     *
     * Когда починено:
     * Локи не висят бесконечно, обрываются с таймаутом lock-wait-timeout
     * Тем самым, пул освобождается в худшем случае за этот таймаут,
     * соответственно последующие запросы будут обработаны.
     */
    @Test
    fun testServiceFailure() {
        mockTvm()

        // bigger than sql_timeout to see connection exceptions when the pool is fully acquired
        val sleepyTime = 40_000L
        // сгенерируем множество параллельных запросов
        Flux.fromIterable(0 until (sqlPoolSize.toInt() + 1)).flatMap {
            WebClient.create("http://localhost:$port").get()
                .uri {
                    it.path("sleep")
                        .queryParam("duration", sleepyTime)
                        .queryParam("lock", true)
                        .build()
                }
                .header("Accept", "application/vnd.api+json")
                .exchangeToMono { it.toEntity(Object::class.java) }
                .toFlux()
        }
            .collectList()
            .timeout(Duration.ofSeconds(1))
            .onErrorResume(ignoreTimeouts())
            .block()

        // check whether service is fully out of order
        val successRequests = Flux.fromIterable(0 until (sqlPoolSize.toInt() + 1)).flatMap {
            WebClient.create("http://localhost:$port").get()
                .uri {
                    it.path("sleep")
                        .queryParam("duration", 1L)
                        .queryParam("lock", false)
                        .build()
                }
                .header("Accept", "application/vnd.api+json")
                .exchangeToMono { it.toEntity(Object::class.java) }
                .toFlux()
        }.collectList()
            // should be bigger than current lock-wait-timeout for test to pass
            .timeout(Duration.ofSeconds(sleepyTime))
            .onErrorResume(ignoreTimeouts())
            .block()

        assertThat(successRequests).allMatch {
            it.statusCode.is2xxSuccessful
        }
    }

    private fun ignoreTimeouts(): (t: Throwable) -> Mono<out MutableList<ResponseEntity<Object>>> =
        {
            if (it is TimeoutException) {
                Mono.empty()
            } else Mono.error(it)
        }

    private fun mockTvm() {
        Mockito.doReturn(
            UserAuthenticationHolder(
                TvmUserAuthentication(
                    true,
                    AuthenticationMethod.AUTH_VIA_TVM_SERVICE,
                    UserCredentials(337625302L)
                )
            )
        )
            .`when`(tvmAuthenticationProvider)
            .authenticate(ArgumentMatchers.any(Authentication::class.java))

        Mockito.doReturn(true)
            .`when`(tvmAuthenticationProvider)
            .supports(ArgumentMatchers.any())

        Mockito.doReturn(AuthenticationType.TVM)
            .`when`(tvmAuthenticationProvider)
            .supportedAuthType()
    }

    class Config {
        @RestController
        class SleepyController(
            private val userService: UserService
        ) {
            @PartnerTransactional
            @GetMapping("/sleep")
            fun sleep(duration: Long, lock: Boolean): Boolean {
                userService.findAll(
                    QueryOpts.forClass(User::class.java)
                        .withFilterByIds(listOf(0L))
                        .forUpdate(lock)
                )
                Thread.sleep(duration)
                return true
            }
        }

        @Auth(AuthenticationType.TVM)
        @Bean
        fun sleepyController(userService: UserService) = SleepyController(userService)
    }
}
