package ru.yandex.partner.jsonapi.i18n;

import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;

import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.springframework.security.core.context.SecurityContextHolder;

import ru.yandex.partner.jsonapi.configuration.locale.PartnerLocaleResolver;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.auth.model.UserAuthenticationHolder;
import ru.yandex.partner.libs.extservice.blackbox.BlackboxUserInfo;

import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mock;

public class PartnerLocaleResolverTest {

    private PartnerLocaleResolver localeResolver;

    @BeforeEach
    public void init() {
        localeResolver = new PartnerLocaleResolver();
        localeResolver.setDefaultLocale(new Locale("ru"));
        localeResolver.setSupportedLocales(List.of("ru", "en", "tr").stream().map(Locale::new)
                .collect(Collectors.toList()));
    }

    @AfterEach
    public void cleanup() {
        SecurityContextHolder.getContext().setAuthentication(null);
    }

    @ParameterizedTest()
    @MethodSource("langSets")
    public void testResolve(String queryLang, String userLang, String headerLang, String expectedLang) {
        HttpServletRequest request = setupRequest(queryLang, userLang, headerLang);
        Assertions.assertEquals(new Locale(expectedLang), localeResolver.resolveLocale(request));
    }

    public static List<Arguments> langSets() {
        return List.of(
                Arguments.of(null, null, null, "ru"),
                Arguments.of(null, null, "tr", "tr"),
                Arguments.of(null, null, "de", "ru"),
                Arguments.of(null, "en", null, "en"),
                Arguments.of(null, "de", null, "ru"),
                Arguments.of(null, "en", "tr", "en"),
                Arguments.of(null, "de", "tr", "tr"),
                Arguments.of(null, "de", "es", "ru"),
                Arguments.of("en", null, null, "en"),
                Arguments.of("de", null, null, "ru"),
                Arguments.of("en", null, "tr", "en"),
                Arguments.of("de", null, "tr", "tr"),
                Arguments.of("de", null, "es", "ru"),
                Arguments.of("en", "ru", "tr", "en"),
                Arguments.of("de", "en", "tr", "en"),
                Arguments.of("de", "es", "tr", "tr"),
                Arguments.of("de", "es", "lt", "ru")
        );
    }

    private HttpServletRequest setupRequest(String queryLang, String userLang, String headerLang) {
        HttpServletRequest request = mock(HttpServletRequest.class);
        doReturn(queryLang).when(request).getParameter("lang");

        UserAuthenticationHolder userAuthenticationHolder = null;
        if (userLang != null) {
            BlackboxUserInfo blackboxUserInfo = mock(BlackboxUserInfo.class);
            doReturn(userLang).when(blackboxUserInfo).getLanguage();
            UserAuthentication userAuthentication = mock(UserAuthentication.class);
            doReturn(blackboxUserInfo).when(userAuthentication).getBlackboxUserInfo();
            userAuthenticationHolder = new UserAuthenticationHolder(userAuthentication);
        }
        SecurityContextHolder.getContext().setAuthentication(userAuthenticationHolder);

        if (headerLang != null) {
            doReturn(headerLang).when(request).getHeader("Accept-Language");
            Locale headerLocale = new Locale(headerLang);
            Enumeration<Locale> locales = Collections.enumeration(List.of(headerLocale));
            doReturn(headerLocale).when(request).getLocale();
            doReturn(locales).when(request).getLocales();
        }

        return request;
    }

}
