package ru.yandex.partner.jsonapi.jackson;

import java.io.IOException;
import java.util.Collections;
import java.util.List;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import org.apache.commons.io.IOUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.partner.core.action.exception.DefectInfoWithMsgParams;
import ru.yandex.partner.core.validation.defects.DefectInfoBuilder;
import ru.yandex.partner.core.validation.defects.presentation.CommonValidationMsg;
import ru.yandex.partner.jsonapi.messages.CommonMsg;
import ru.yandex.partner.jsonapi.models.ModelClassUtils;
import ru.yandex.partner.libs.i18n.MsgWithArgs;

class CollectingErrorsParserTest {

    @Test
    void parse() throws IOException {
        var collectingErrorsParser = new CollectingErrorsParser();
        String json = getResourceData("test_model.json");
        JsonNode jsonNode = ObjectMapperFactory.createObjectMapper().readTree(json);
        JavaType type = ModelClassUtils.valType(TestModel.class);
        Object object = collectingErrorsParser.parse(jsonNode, type);
        Assertions.assertEquals(getExpectedTestModel(), object);
        var defects = collectingErrorsParser.getAndClearCollectedDefects("test_model");
        Assertions.assertEquals(Collections.emptyList(), defects);
    }

    @Test
    void parseWithDefects() throws IOException {
        var collectingErrorsParser = new CollectingErrorsParser();
        String json = getResourceData("test_model_with_errors.json");
        JsonNode jsonNode = ObjectMapperFactory.createObjectMapper().readTree(json);
        JavaType type = ModelClassUtils.valType(TestModel.class);
        Object object = collectingErrorsParser.parse(jsonNode, type);
        Assertions.assertNotNull(object);
        var defects = collectingErrorsParser.getAndClearCollectedDefects("test_model");
        Assertions.assertEquals(getExpectedDefects(), defects);
    }

    @Test
    void parseWithDefect() throws IOException {
        var collectingErrorsParser = new CollectingErrorsParser();
        String json = getResourceData("test_model_with_error.json");
        JsonNode jsonNode = ObjectMapperFactory.createObjectMapper().readTree(json);
        Object object = collectingErrorsParser.parse(jsonNode,
                ModelClassUtils.getListClass(TestModel.class).getJavaType());
        Assertions.assertNotNull(object);

        var defects = collectingErrorsParser.getAndClearCollectedDefects("test_model");
        Assertions.assertEquals(1, defects.size());
        var defect = defects.get(0).getDefect().params();
        Assertions.assertTrue(defect instanceof DefectInfoWithMsgParams);
        Assertions.assertEquals(MsgWithArgs.of(CommonValidationMsg.UNKNOWN_KEY, "not_exists"),
                ((DefectInfoWithMsgParams) defect).getGettextMsg());
    }

    private TestModel getExpectedTestModel() {
        var testModel = new TestModel();
        testModel.setId(1);
        testModel.setName("name");
        testModel.setChildren(List.of(2, 3));

        var attributes = List.of(
                new TestModelAttribute().withId(0),
                new TestModelAttribute().withName("kyky"),
                new TestModelAttribute().withId(2).withName("full")
        );

        testModel.setAttributes(attributes);

        var testModelAttribute = new TestModelAttribute();
        testModelAttribute.setId(34);
        testModelAttribute.setName("some_name");
        testModel.setAttribute1(testModelAttribute);

        testModel.setAttribute2(new TestModelAttribute());
        return testModel;
    }

    private List<DefectInfo<Defect>> getExpectedDefects() {
        return List.of(
                DefectInfoBuilder.of(CommonValidationMsg.DATA_MUST_BE_INTEGER_NUMBER)
                        .withPath(new Path(List.of(
                                new PathNode.Field("test_model"),
                                new PathNode.Field("id")
                        )))
                        .build(),
                DefectInfoBuilder.of(CommonMsg.DATA_MUST_BE_STRING)
                        .withPath(new Path(List.of(
                                new PathNode.Field("test_model"),
                                new PathNode.Field("name")
                        ))).build(),
                DefectInfoBuilder.of(CommonMsg.DATA_MUST_BE_ARRAY)
                        .withPath(new Path(List.of(
                                new PathNode.Field("test_model"),
                                new PathNode.Field("children")
                        )))
                        .build(),
                DefectInfoBuilder.of(CommonValidationMsg.DATA_MUST_BE_INTEGER_NUMBER)
                        .withPath(new Path(List.of(
                                new PathNode.Field("test_model"),
                                new PathNode.Field("attributes"),
                                new PathNode.Field("1"),
                                new PathNode.Field("id")
                        )))
                        .build(),
                DefectInfoBuilder.of(CommonMsg.DATA_MUST_BE_STRING)
                        .withPath(new Path(List.of(
                                new PathNode.Field("test_model"),
                                new PathNode.Field("attributes"),
                                new PathNode.Field("2"),
                                new PathNode.Field("name")
                        )))
                        .build(),
                DefectInfoBuilder.of(MsgWithArgs.of(CommonValidationMsg.UNKNOWN_KEY, "unknown_array"))
                        .withPath(new Path(List.of(
                                new PathNode.Field("test_model")
                        )))
                        .build(),
                DefectInfoBuilder.of(CommonMsg.DATA_MUST_BE_HASH)
                        .withPath(new Path(List.of(
                                new PathNode.Field("test_model"),
                                new PathNode.Field("attribute1")
                        )))
                        .build(),
                DefectInfoBuilder.of(CommonMsg.DATA_MUST_BE_HASH)
                        .withPath(new Path(List.of(
                                new PathNode.Field("test_model"),
                                new PathNode.Field("attribute2")
                        )))
                        .build(),
                DefectInfoBuilder.of(MsgWithArgs.of(CommonValidationMsg.UNKNOWN_KEY, "unknown_object"))
                        .withPath(new Path(List.of(
                                new PathNode.Field("test_model")
                        )))
                        .build(),
                DefectInfoBuilder.of(CommonValidationMsg.DATA_MUST_BE_INTEGER_NUMBER)
                        .withPath(new Path(List.of(
                                new PathNode.Field("test_model"),
                                new PathNode.Field("attribute3"),
                                new PathNode.Field("id")
                        )))
                        .build(),
                DefectInfoBuilder.of(MsgWithArgs.of(CommonValidationMsg.UNKNOWN_KEY, "unknown_tag"))
                        .withPath(new Path(List.of(
                                new PathNode.Field("test_model"),
                                new PathNode.Field("attribute3")
                        )))
                        .build()
        );
    }

    private String getResourceData(String fileName) throws IOException {
        return IOUtils.toString(
                this.getClass()
                        .getResourceAsStream(
                                "/".concat(getClass().getPackageName().replace('.', '/'))
                                        .concat("/")
                                        .concat(fileName)
                        )
        );
    }
}
