package ru.yandex.partner.jsonapi.multistate;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Nonnull;

import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.entity.block.multistate.InternalMobileRtbBlockActionChecksService;
import ru.yandex.partner.core.entity.block.multistate.InternalMobileRtbBlockMultistateGraph;
import ru.yandex.partner.core.entity.block.multistate.InternalRtbBlockActionChecksService;
import ru.yandex.partner.core.entity.block.multistate.InternalRtbBlockMultistateGraph;
import ru.yandex.partner.core.entity.block.multistate.MobileRtbBlockActionChecksService;
import ru.yandex.partner.core.entity.block.multistate.MobileRtbBlockMultistateGraph;
import ru.yandex.partner.core.entity.block.multistate.RtbBlockActionChecksService;
import ru.yandex.partner.core.entity.block.multistate.RtbBlockMultistateGraph;
import ru.yandex.partner.core.entity.block.type.siteversionandcontextpage.SiteVersionAvailabilityFacade;
import ru.yandex.partner.core.entity.block.type.siteversionandcontextpage.SiteVersionAvailabilityService;
import ru.yandex.partner.core.entity.page.multistate.ContextPageMultistateGraph;
import ru.yandex.partner.core.entity.page.multistate.InternalContextPageMultistateGraph;
import ru.yandex.partner.core.entity.page.multistate.InternalMobileAppMultistateGraph;
import ru.yandex.partner.core.entity.page.multistate.MobileAppMultistateGraph;
import ru.yandex.partner.core.entity.queue.multistate.TaskMultistateGraph;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.entity.user.multistate.CanChangeRequisitesCheck;
import ru.yandex.partner.core.entity.user.multistate.UserActionChecksService;
import ru.yandex.partner.core.entity.user.multistate.UserMultistateGraph;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.StateFlag;
import ru.yandex.partner.core.multistate.block.BlockMultistate;
import ru.yandex.partner.core.multistate.page.InternalMobileAppMultistate;
import ru.yandex.partner.core.multistate.queue.TaskMultistate;
import ru.yandex.partner.core.multistate.user.UserMultistate;
import ru.yandex.partner.jsonapi.crnk.action.check.ActionChecker;
import ru.yandex.partner.jsonapi.crnk.action.check.UserSimpleInappActionChecker;
import ru.yandex.partner.jsonapi.crnk.authorization.actions.AbstractActionsAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.block.authorization.actions.BlockActionsCheckers;
import ru.yandex.partner.jsonapi.crnk.block.authorization.actions.check.BlockIsGodModeChecker;
import ru.yandex.partner.jsonapi.crnk.block.authorization.actions.check.BlockNotReadOnlyOrUserCanEditReadOnlyChecker;
import ru.yandex.partner.jsonapi.crnk.block.authorization.actions.check.CanAddBlockChecker;
import ru.yandex.partner.jsonapi.crnk.block.authorization.actions.check.MobileRtbNotReadOnlyAssistantChecker;
import ru.yandex.partner.jsonapi.crnk.block.authorization.actions.check.RtbNotReadOnlyAssistantChecker;
import ru.yandex.partner.jsonapi.crnk.block.rtb.external.authorization.actions.RtbBlockActionsAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.block.rtb.external.authorization.actions.RtbBlockActionsCheckers;
import ru.yandex.partner.jsonapi.crnk.block.rtb.internal.authorization.actions.InternalRtbBlockActionsAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.block.rtb.internal.authorization.actions.InternalRtbBlockActionsCheckers;
import ru.yandex.partner.jsonapi.crnk.block.rtb.mobile.external.MobileRtbBlockActionsAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.block.rtb.mobile.external.authorization.actions.MobileRtbBlockActionsCheckers;
import ru.yandex.partner.jsonapi.crnk.block.rtb.mobile.internal.authorization.actions.InternalMobileRtbBlockActionsAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.block.rtb.mobile.internal.authorization.actions.InternalMobileRtbBlockActionsCheckers;
import ru.yandex.partner.jsonapi.crnk.page.PageComputableApiFieldsService;
import ru.yandex.partner.jsonapi.crnk.user.authorization.actions.UserActionsAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.user.authorization.actions.check.CanChangeContractUserChecker;
import ru.yandex.partner.jsonapi.crnk.user.authorization.actions.check.CanEditAllOrIsCurrentUserChecker;
import ru.yandex.partner.jsonapi.crnk.user.authorization.actions.check.IsCurrentUserChecker;
import ru.yandex.partner.jsonapi.crnk.user.authorization.actions.check.UserCheckerComposite;
import ru.yandex.partner.jsonapi.models.block.rtb.external.ApiRtbMetaData;
import ru.yandex.partner.jsonapi.models.block.rtb.internal.ApiInternalRtbMetaData;
import ru.yandex.partner.jsonapi.models.block.rtb.mobile.external.ApiMobileRtbMetaData;
import ru.yandex.partner.jsonapi.models.block.rtb.mobile.internal.ApiInternalMobileRtbMetaData;
import ru.yandex.partner.jsonapi.models.user.ApiUserModelMetaData;
import ru.yandex.partner.jsonapi.multistate.dto.MultistateConfigJson;
import ru.yandex.partner.jsonapi.multistate.dto.StateFlagOptsJson;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;
import ru.yandex.partner.test.utils.TestUtils;

import static java.util.function.Function.identity;
import static org.mockito.Mockito.mock;

public class MultistateConfigurationTest {

    protected static <T> Map<String, Integer> calculateBits(
            List<Optional<T>> flags) {
        return IntStream.range(0, flags.size()).boxed()
                .collect(Collectors.toMap(
                        i -> getStateName(flags.get(i)),
                        identity(),
                        (v1, v2) -> v1, TreeMap::new
                ));
    }

    @Nonnull
    private static <T> String getStateName(Optional<T> optionalFlag) {
        return optionalFlag.map(
                f -> ((Enum<?>) f).name().toLowerCase()
        ).orElse("free_multistate");
    }

    protected static <T extends StateFlag> Map<String, StateFlagOptsJson> calculateOpts(
            List<Optional<T>> flags) {
        return flags.stream()
                .collect(Collectors.toMap(
                        MultistateConfigurationTest::getStateName,
                        MultistateConfigurationTest::calculateOptsForFlag
                ))
                .entrySet().stream()
                .filter(e -> e.getValue().isPresent())
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        e -> e.getValue().get(),
                        (v1, v2) -> v1, TreeMap::new
                ));
    }

    @Nonnull
    private static <T extends StateFlag> Optional<StateFlagOptsJson> calculateOptsForFlag(Optional<T> flag) {
        return (flag.isEmpty() || flag.get().isPrivate()) ? Optional.of(new StateFlagOptsJson(1)) : Optional.empty();
    }

    @Test
    public void userMultistateConfigTest() throws Exception {
        UserMultistateGraph userMultistateGraph =
                new UserMultistateGraph(new UserActionChecksService(mock(CanChangeRequisitesCheck.class)));

        var currentUserChecker = new IsCurrentUserChecker<User>();
        List<ActionChecker<User>> checks = List.of(
                currentUserChecker,
                new CanEditAllOrIsCurrentUserChecker<>(currentUserChecker),
                new CanChangeContractUserChecker<>()
        );

        var userCheckerComposite = new UserCheckerComposite<>(checks);

        UserActionsAuthorizationService userActionsAuthorizationService = new UserActionsAuthorizationService(
                userMultistateGraph,
                new ApiUserModelMetaData(),
                mock(AuthenticationFacade.class),
                userCheckerComposite);

        testConfig(
                userMultistateGraph, userActionsAuthorizationService,
                UserMultistate.getStateFlagsInOrder(), "users.json"
        );
    }

    @Test
    public void rtbBlockMultistateConfigTest() throws Exception {
        var rtbBlockActionChecksService = new RtbBlockActionChecksService(mock(ContextPageMultistateGraph.class),
                new SiteVersionAvailabilityService(mock(SiteVersionAvailabilityFacade.class)));

        RtbBlockMultistateGraph blockMultistateGraph = new RtbBlockMultistateGraph(rtbBlockActionChecksService);

        BlockActionsCheckers<RtbBlock> blockActionsCheckers = new RtbBlockActionsCheckers(
                new BlockNotReadOnlyOrUserCanEditReadOnlyChecker(),
                new RtbNotReadOnlyAssistantChecker(Mockito.mock(PageComputableApiFieldsService.class)),
                new BlockIsGodModeChecker(),
                new CanAddBlockChecker<>()
        );

        RtbBlockActionsAuthorizationService blockActionsAuthorizationService =
                new RtbBlockActionsAuthorizationService(
                        new ApiRtbMetaData(),
                        blockMultistateGraph, mock(AuthenticationFacade.class),
                        blockActionsCheckers);

        testConfig(
                blockMultistateGraph, blockActionsAuthorizationService,
                BlockMultistate.getStateFlagsInOrder(), "context_on_site_rtb.json"
        );
    }

    @Test
    public void internalRtbBlockMultistateConfigTest() throws Exception {
        var rtbBlockActionChecksService =
                new InternalRtbBlockActionChecksService(mock(InternalContextPageMultistateGraph.class),
                        mock((SiteVersionAvailabilityService.class)));

        var blockMultistateGraph = new InternalRtbBlockMultistateGraph(rtbBlockActionChecksService);

        var blockActionsCheckers = new InternalRtbBlockActionsCheckers(
                new BlockIsGodModeChecker(),
                new CanAddBlockChecker<>()
        );

        var blockActionsAuthorizationService =
                new InternalRtbBlockActionsAuthorizationService(
                        new ApiInternalRtbMetaData(),
                        blockMultistateGraph, mock(AuthenticationFacade.class),
                        blockActionsCheckers);

        testConfig(
                blockMultistateGraph, blockActionsAuthorizationService,
                BlockMultistate.getStateFlagsInOrder(), "internal_context_on_site_rtb.json"
        );
    }

    @Test
    public void mobileRtbBlockMultistateConfigTest() throws Exception {
        var rtbBlockActionChecksService =
                new MobileRtbBlockActionChecksService(mock(MobileAppMultistateGraph.class));

        var blockMultistateGraph = new MobileRtbBlockMultistateGraph(rtbBlockActionChecksService);

        var blockActionsCheckers = new MobileRtbBlockActionsCheckers(mock(UserSimpleInappActionChecker.class),
                mock(CanAddBlockChecker.class), mock(BlockIsGodModeChecker.class),
                mock(MobileRtbNotReadOnlyAssistantChecker.class),
                new BlockNotReadOnlyOrUserCanEditReadOnlyChecker());

        var blockActionsAuthorizationService = new MobileRtbBlockActionsAuthorizationService(
                blockMultistateGraph,
                new ApiMobileRtbMetaData(),
                mock(AuthenticationFacade.class),
                blockActionsCheckers
        );

        testConfig(
                blockMultistateGraph, blockActionsAuthorizationService,
                BlockMultistate.getStateFlagsInOrder(), "mobile_app_rtb.json"
        );
    }

    @Test
    public void internalMobileAppMultistateConfigTest() throws Exception {
        var internalMobileAppMultistateGraph = new InternalMobileAppMultistateGraph();

        testConfig(
                internalMobileAppMultistateGraph, null,
                InternalMobileAppMultistate.getStateFlagsInOrder(), "internal_mobile_app.json"
        );
    }

    @Test
    public void internalMobileRtbBlockMultistateConfigTest() throws Exception {
        var rtbBlockActionChecksService =
                new InternalMobileRtbBlockActionChecksService(mock(InternalMobileAppMultistateGraph.class));

        var blockMultistateGraph = new InternalMobileRtbBlockMultistateGraph(rtbBlockActionChecksService);

        var blockActionsCheckers = new InternalMobileRtbBlockActionsCheckers(mock(BlockIsGodModeChecker.class),
                mock(CanAddBlockChecker.class));

        var blockActionsAuthorizationService = new InternalMobileRtbBlockActionsAuthorizationService(
                new ApiInternalMobileRtbMetaData(),
                blockMultistateGraph,
                mock(AuthenticationFacade.class),
                blockActionsCheckers
        );

        testConfig(
                blockMultistateGraph, blockActionsAuthorizationService,
                BlockMultistate.getStateFlagsInOrder(), "internal_mobile_app_rtb.json"
        );
    }

    @Test
    public void taskMultistateConfigTest() throws Exception {
        TaskMultistateGraph taskMultistateGraph = new TaskMultistateGraph();

        testConfig(
                taskMultistateGraph, null,
                TaskMultistate.getStateFlagsInOrder(), "queue.json"
        );
    }

    protected <M, C, T extends StateFlag> void testConfig(
            MultistateGraph<M, T> multistateGraph,
            AbstractActionsAuthorizationService<? extends M> actionsAuthorizationService,
            List<Optional<T>> stateFlagsInOrder,
            String fileName
    ) throws Exception {
        Set<String> actions = new TreeSet<>(multistateGraph.getAllAvailableActionNames());
        Map<String, Integer> bits = calculateBits(stateFlagsInOrder);
        Map<String, StateFlagOptsJson> opts = calculateOpts(stateFlagsInOrder);

        Map<String, Map<String, Long>> graph = calculateGraph(multistateGraph.getAllTransitions());

        Map<String, String> rights = calculateRights(actionsAuthorizationService);

        MultistateConfigJson multistateConfigJson = new MultistateConfigJson(actions, bits, graph, opts, rights);

        TestUtils.compareToDataFromFile(multistateConfigJson,
                MultistateConfigurationTest.class, fileName);
    }

    @Nonnull
    protected <T extends StateFlag> Map<String, Map<String, Long>> calculateGraph(
            Map<Multistate<T>, Map<String, Multistate<T>>> allTransitions) {
        return allTransitions.entrySet().stream()
                .collect(Collectors.toMap(
                        e -> String.valueOf(e.getKey().toMultistateValue()),
                        e -> calculateGraphEntry(e.getValue()),
                        (v1, v2) -> v1, TreeMap::new
                ));
    }

    @Nonnull
    private <T extends StateFlag> Map<String, Long> calculateGraphEntry(
            Map<String, Multistate<T>> transitions) {
        return transitions.entrySet().stream()
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        e1 -> e1.getValue().toMultistateValue(),
                        (v1, v2) -> v1, TreeMap::new
                ));
    }

    @Nonnull
    private <M> Map<String, String> calculateRights(
            AbstractActionsAuthorizationService<M> actionsAuthorizationService
    ) {
        return actionsAuthorizationService == null ? new TreeMap<>()
                : new TreeMap<>(actionsAuthorizationService.getRightNamesForActions());
    }

}
