package ru.yandex.partner.jsonapi.sql;

import java.io.IOException;
import java.util.function.Function;
import java.util.stream.Stream;

import org.junit.jupiter.api.extension.ExtensionContext;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.ArgumentsProvider;
import org.junit.platform.commons.util.StringUtils;

class SqlArgumentProvider implements ArgumentsProvider {
    private static final String DEFAULT_SOURCE_PATH = "/sql";

    @Override
    public Stream<? extends Arguments> provideArguments(ExtensionContext context) throws Exception {
        String sourcePath = getSourcePath();

        return Stream.of(
                        getInternalRtb(),
                        getMobileRtb(),
                        getInternalMobileRtb(),
                        getRtbSql(),
                        getUserSql()
                )
                .flatMap(Function.identity())
                .filter(sqlArgument -> sqlArgument.getResourcePath().startsWith(sourcePath))
                .map(Arguments::of);
    }

    private String getSourcePath() throws IOException {
        String sqlSourcePath = System.getenv("SQL_SOURCE_PATH");
        String path = "/jsonapi/src/test/resources" + DEFAULT_SOURCE_PATH;
        if (StringUtils.isBlank(sqlSourcePath)) {
            return DEFAULT_SOURCE_PATH;
        } else if (sqlSourcePath.contains(path)) {
            return sqlSourcePath.substring(sqlSourcePath.lastIndexOf(DEFAULT_SOURCE_PATH));
        } else {
            throw new IOException("Environment variable SQL_SOURCE_PATH is incorrect. It has to contain path: " +
                    path);
        }
    }

    private Stream<SqlArgument> getUserSql() {
        return Stream.of(
                new HttpGetSqlArgument("/v1/users/1015",
                        "mocked-yan-manager",
                        "/sql/users/mocked-yan-manager/get"),
                new HttpGetSqlArgument("/v1/users/1015/?fields[users]=login,avatar,notifications_count,roles",
                        "mocked-yan-manager",
                        "/sql/users/mocked-yan-manager/fields"),
                new HttpGetSqlArgument("/v1/users/1009",
                        "mocked-yan-partner",
                        "/sql/users/mocked-yan-partner/get"),
                new HttpGetSqlArgument("/v1/users/?filter=[\"id\", \"IN\", [0, 1009, 1008]]",
                        "mocked-yan-manager",
                        "/sql/users/mocked-yan-manager/authorization-filter"),
                new HttpGetSqlArgument("/v1/users/?fields[users]=id&page[size]=3&meta=total",
                        "mocked-yan-manager",
                        "/sql/users/mocked-yan-manager/get-all-limited"),
                new HttpGetSqlArgument("/v1/users/",
                        "tutby-cop",
                        "/sql/users/tutby-cop/authorization-filter"),
                new HttpGetSqlArgument("/v1/users/",
                        "mocked-yan-partner",
                        "/sql/users/mocked-yan-partner/authorization-filter"),
                new HttpGetSqlArgument("/v1/users/",
                        "mocked-yan-partner-assistant",
                        "/sql/users/mocked-yan-partner-assistant/authorization-filter"),
                new HttpGetSqlArgument("/v1/users/?fields[users]=id&page[size]=3&meta=total",
                        "mocked-yan-partner-assistant",
                        "/sql/users/mocked-yan-partner-assistant/get-all-limited")
        );
    }

    private Stream<SqlArgument> getRtbSql() {
        return Stream.of(
                new HttpGetSqlArgument("/v1/context_on_site_rtb/R-A-41443-1?$ALL_FIELDS[context_on_site_rtb]",
                        "mocked-yan-partner",
                        "/sql/block/rtb/external/mocked-yan-partner/get"),
                new HttpGetSqlArgument("/v1/context_on_site_rtb/R-A-41443-1?$ALL_FIELDS[context_on_site_rtb]",
                        "mocked-yan-manager",
                        "/sql/block/rtb/external/mocked-yan-manager/get"),
                new HttpGetSqlArgument("/v1/context_on_site_rtb/R-A-41443-1?$ALL_FIELDS[context_on_site_rtb]",
                        "mocked-yan-partner-assistant",
                        "/sql/block/rtb/external/mocked-yan-partner-assistant/get"),
                new HttpGetSqlArgument("/v1/context_on_site_rtb/R-A-41443-1",
                        "mocked-robot-assistant",
                        "/sql/block/rtb/external/mocked-robot-assistant/get"),
                new HttpGetSqlArgument("/v1/context_on_site_rtb/R-A-41443-1?fields[context_on_site_rtb]=actions",
                        "mocked-yan-manager",
                        "/sql/block/rtb/external/mocked-yan-manager/get2"),
                new HttpGetSqlArgument(
                        "/v1/context_on_site_rtb?fields=id&filter=[\"campaign\",\"MATCH\"," +
                                "[\"all_domain\",\"=\",\"showmethemoney.ru\"]]",
                        "mocked-yan-partner",
                        "/sql/block/rtb/external/mocked-yan-partner/get-all-domain"),
                new HttpGetSqlArgument(
                        "/v1/context_on_site_rtb?fields=id&filter=[\"campaign.all_domain\"," +
                                "\"=\",\"showmethemoney.ru\"]]",
                        "mocked-yan-partner",
                        "/sql/block/rtb/external/mocked-yan-partner/get-all-domain_2"),
                new HttpGetSqlArgument("/v1/context_on_site_rtb?fields=site_version" +
                        "&filter=[\"site_version\",\"=\",\"general\"]",
                        "mocked-yan-partner",
                        "/sql/block/rtb/external/mocked-yan-partner/get-site-version"),
                new HttpGetSqlArgument(
                        "/v1/context_on_site_rtb?fields[context_on_site_rtb]=page_id,id&filter=[\"id\",\"IN\"," +
                                "[\"R-A-41443-1\"]]",
                        "mocked-yan-partner",
                        "/sql/block/rtb/external/mocked-yan-partner/filter-public-id"),
                new HttpGetSqlArgument(
                        "/v1/context_on_site_rtb?fields[context_on_site_rtb]=page_id,id&filter=[\"id\",\"IN\"," +
                                "[\"41443-1\"]]",
                        "mocked-yan-partner",
                        "/sql/block/rtb/external/mocked-yan-partner/filter-public-id-wo-prefix"),
                new HttpGetSqlArgument(
                        "/v1/context_on_site_rtb?fields[context_on_site_rtb]=page_id,id&filter=[\"id\",\"IN\"," +
                                "[\"9\"]]",
                        "mocked-yan-partner",
                        "/sql/block/rtb/external/mocked-yan-partner/filter-public-id-only-block-id"),
                new HttpGetSqlArgument(
                        "/v1/context_on_site_rtb?fields[context_on_site_rtb]=page_id,id&filter=[\"id\",\"IN\"," +
                                "[\"invalid-value\"]]",
                        "mocked-yan-partner",
                        "/sql/block/rtb/external/mocked-yan-partner/filter-public-id-invalid-value"),
                new HttpGetSqlArgument(
                        "/v1/context_on_site_rtb/?$ALL_FIELDS[context_on_site_rtb]",
                        "tutby-cop",
                        "/sql/block/rtb/external/tutby-cop/get-all"),


                new HttpGetSqlArgument(
                        "/v1/kv_store_frontend?filter=[\"user_id\", \"IN\", [1009]]&$ALL_FIELDS[kv_store_frontend]",
                        "mocked-yan-partner",
                        "/sql/kv_store_frontend/mocked-yan-partner/get-current"),
                new HttpGetSqlArgument(
                        "/v1/kv_store_frontend?filter=[\"user_id\", \"IN\", [1015]]&$ALL_FIELDS[kv_store_frontend]",
                        "mocked-yan-manager",
                        "/sql/kv_store_frontend/mocked-yan-manager/get-current"),
                new HttpPostSqlArgument("/v1/context_on_site_rtb/R-A-41443-1/action/edit",
                        "mocked-yan-manager",
                        "/sql/block/rtb/external/mocked-yan-manager/edit",
                        "{\"caption\": \"updated caption\"}"),
                new HttpPostSqlArgument("/v1/context_on_site_rtb/R-A-41443-1/action/duplicate",
                        "mocked-yan-manager",
                        "/sql/block/rtb/external/mocked-yan-manager/duplicate",
                        "")
        );
    }

    private Stream<SqlArgument> getInternalRtb() {
        return Stream.of(
                new HttpGetSqlArgument(
                        "/v1/internal_context_on_site_rtb/R-A-41443-10?$ALL_FIELDS[context_on_site_rtb]",
                        "yndx-developer",
                        "/sql/block/rtb/internal/yndx-developer/get"
                ),
                new HttpGetSqlArgument(
                        "/v1/internal_context_on_site_rtb/?filter=[\"campaign.all_domain\", \"LIKE\",\"test\"]",
                        "mocked-internal-administrator",
                        "/sql/block/rtb/internal/mocked-internal-administrator/domain_filter"
                )
        );
    }

    private Stream<SqlArgument> getMobileRtb() {
        return Stream.of(
                new HttpGetSqlArgument(
                        "/v1/mobile_app_rtb?fields[mobile_app_rtb]=caption&filter=[\"is_mobile_mediation\", \"=\", 1]",
                        "yndx-developer",
                        "/sql/block/mobile-rtb/yndx-developer/check_filter_by_json_path"),
                new HttpGetSqlArgument("/v1/mobile_app_rtb/R-M-43569-1?$ALL_FIELDS[mobile_app_rtb]",
                        "mocked-mobile-app-partner",
                        "/sql/block/rtb/mobile/mocked-yan-partner/get"),
                new HttpGetSqlArgument("/v1/mobile_app_rtb/R-M-43569-1?$ALL_FIELDS[mobile_app_rtb]",
                        "mocked-yan-manager",
                        "/sql/block/rtb/mobile/mocked-yan-manager/get")
        );
    }

    private Stream<SqlArgument> getInternalMobileRtb() {
        return Stream.of(
                new HttpGetSqlArgument(
                        "/v1/internal_mobile_app_rtb/?*/drop table context_on_site_rtb/*",
                        "yndx-developer",
                        "/sql/block/rtb/internal/mobile/yndx-developer/sql_injection1"),
                new HttpGetSqlArgument(
                        "/v1/internal_mobile_app_rtb/R-IM-132439-1/actions*/drop table context_on_site_rtb -- hello",
                        "yndx-developer",
                        "/sql/block/rtb/internal/mobile/yndx-developer/sql_injection2"),
                new HttpGetSqlArgument(
                        "/v1/internal_mobile_app_rtb?fields[internal_mobile_app_rtb]=caption",
                        "yndx-developer",
                        "/sql/block/rtb/internal/mobile/yndx-developer/check_filter_by_json_path"),
                new HttpGetSqlArgument("/v1/internal_mobile_app_rtb/R-IM-132439-1?$ALL_FIELDS[internal_mobile_app_rtb]",
                        "mocked-internal-administrator",
                        "/sql/block/rtb/internal/mobile/mocked-internal-administrator/get"),
                new HttpGetSqlArgument("/v1/internal_mobile_app_rtb/R-IM-132439-1?$ALL_FIELDS[internal_mobile_app_rtb]",
                        "mocked-internal-manager",
                        "/sql/block/rtb/internal/mobile/mocked-internal-manager/get"),
                new HttpGetSqlArgument("/v1/internal_mobile_app_rtb/R-IM-132439-1?$ALL_FIELDS[internal_mobile_app_rtb]",
                        "mocked-yan-manager",
                        "/sql/block/rtb/internal/mobile/mocked-yan-manager/get"),
                new HttpGetSqlArgument("/v1/internal_mobile_app_rtb/R-IM-132439-1?$ALL_FIELDS[internal_mobile_app_rtb]",
                        "mocked-yan-partner",
                        "/sql/block/rtb/internal/mobile/mocked-yan-partner/get")
        );
    }
}
