package ru.yandex.partner.jsonapi.sql;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Clock;
import java.time.ZoneId;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import io.crnk.core.engine.registry.ResourceRegistry;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ArgumentsSource;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.web.reactive.function.client.WebClient;

import ru.yandex.partner.core.junit.MySqlRefresher;
import ru.yandex.partner.defaultconfiguration.PartnerLocalDateTime;
import ru.yandex.partner.jsonapi.JsonApiTest;
import ru.yandex.partner.jsonapi.configuration.TestQuerySpecUrlMapper;
import ru.yandex.partner.jsonapi.models.ApiService;
import ru.yandex.partner.test.db.QueryLogService;
import ru.yandex.partner.test.utils.TestUtils;

import static org.assertj.core.api.Assertions.assertThat;

// Спсособ сделать методы @BeforeAll и @AfterAll не статичными
@TestInstance(TestInstance.Lifecycle.PER_CLASS)
@ExtendWith(MySqlRefresher.class)
@JsonApiTest
public class SqlTest {
    private Set<String> crnkResourceTypes;

    private WebClient webClient;

    private final DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

    @Autowired
    private TestQuerySpecUrlMapper testQuerySpecUrlMapper;

    @LocalServerPort
    private int port;

    @Autowired
    private QueryLogService queryLogService;

    @Autowired
    private ResourceRegistry resourceRegistry;

    @Autowired
    private List<ApiService<?>> apiServices;

    @BeforeAll
    void beforeAll() throws ParseException {
        crnkResourceTypes = apiServices.stream()
                .map(apiService -> apiService.getApiModel().getResourceType())
                .collect(Collectors.toSet());
        PartnerLocalDateTime.setClock(Clock.fixed(
                dateFormat.parse("2021-12-02 12:13:14").toInstant(),
                ZoneId.systemDefault()));
    }

    @AfterAll
    void afterAll() {
        Assertions.assertTrue(crnkResourceTypes.isEmpty(),
                "There are crnk types that have not been tested: " +
                        Arrays.toString(crnkResourceTypes.toArray())
        );
    }

    @ParameterizedTest
    @ArgumentsSource(SqlArgumentProvider.class)
    void requestResponseTest(SqlArgument sqlArgument) {
        sqlArgument.performRequest(webClient, resourceRegistry);
        String resourcePath = sqlArgument.getResourcePath();
        List<String> actual = queryLogService.getLog();

        if (TestUtils.needSelfUpdate()) {
            TestUtils.doSQLSelfUpdate(actual, resourcePath);
            Assertions.fail("Resource self_updated. Path = " + resourcePath);
        } else {
            List<String> expected = TestUtils.getSqlTestData(resourcePath);
            assertThat(actual).hasSize(expected.size());

            List<String> actualList = TestUtils.normalizeQueries(actual);
            List<String> expectedList = TestUtils.normalizeQueries(expected);

            for (int i = 0; i < expectedList.size(); i++) {
                assertThat(actualList.get(i)).isEqualTo(expectedList.get(i));
            }
        }
        crnkResourceTypes.remove(testQuerySpecUrlMapper.getLastCalledType());
    }

    @BeforeEach
    void setUp() {
        webClient = WebClient.create("http://127.0.0.1:" + port);
    }
}
