package ru.yandex.partner.jsonapi.transactions;

import org.jooq.DSLContext;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Scope;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import ru.yandex.partner.jsonapi.JsonApiTest;
import ru.yandex.partner.libs.annotation.PartnerTransactional;

import static org.assertj.core.api.Assertions.assertThat;
import static ru.yandex.partner.dbschema.partner.Tables.KV_STORE_FRONTEND;

@Disabled
@Import(NestedTransactionDemo.TransactionalBeans.class)
@JsonApiTest
public class NestedTransactionDemo {

    public static final String INITIAL_VALUE = "common_value";
    @Autowired
    private TransactionalBeans transactionalBeans;
    @Autowired
    private DSLContext dslContext;

    @Test
    public void testSuccess() {
        transactionalBeans.transactionalAction(() -> {
            setValue("first");

            transactionalBeans.transactionalAction(() -> {
                setValue("second");
            }).perform();

        }).perform();

        assertThat(getGetValue()).isEqualTo("second");
    }

    @Test
    public void testFullRollbackByException() {
        try {
            transactionalBeans.transactionalAction(() -> {
                setValue("first");

                transactionalBeans.transactionalAction(() -> {
                    setValue("second");

                    transactionalBeans.transactionalAction(() -> {
                        setValue("third");
                        throw new RuntimeException("rollback all transactions");
                    }).perform();

                }).perform();

            }).perform();
        } catch (RuntimeException expected) {
        }

        assertThat(getGetValue()).isEqualTo(INITIAL_VALUE);
    }

    @Test
    public void testFullRollbackNestedByManager() {

        transactionalBeans.transactionalAction(() -> {
            setValue("first");

            transactionalBeans.transactionalAction(() -> {
                setValue("second");

                transactionalBeans.transactionalAction(() -> {
                    setValue("third");
                    TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
                }).perform();

            }).perform();

        }).perform();


        assertThat(getGetValue()).isEqualTo("second");
    }

    @Test
    public void testPartialRollbackWithoutExceptions() {
        transactionalBeans.transactionalAction(() -> {
            setValue("first");

            transactionalBeans.transactionalAction(() -> {
                setValue("second");
                TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            }).perform();

        }).perform();

        assertThat(getGetValue()).isEqualTo("first");
    }

    @Test
    public void testPartialRollbackByException() {
        transactionalBeans.transactionalAction(() -> {
            setValue("first");

            try {
                transactionalBeans.transactionalAction(() -> {
                    setValue("second");
                    throw new RuntimeException("rollback nested transaction");
                }).perform();
            } catch (RuntimeException expected) {

            }
        }).perform();

        assertThat(getGetValue()).isEqualTo("first");
    }

    private String getGetValue() {
        return dslContext.select(KV_STORE_FRONTEND.VALUE).from(KV_STORE_FRONTEND)
                .where(KV_STORE_FRONTEND.ID.eq(1L))
                .fetch(KV_STORE_FRONTEND.VALUE)
                .get(0);
    }

    private int setValue(String value) {
        return dslContext.update(KV_STORE_FRONTEND)
                .set(KV_STORE_FRONTEND.VALUE, value)
                .where(KV_STORE_FRONTEND.ID.eq(1L))
                .execute();
    }

    interface Action {
        void perform();
    }

    @Configuration
    public static class TransactionalBeans {
        @Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
        @Bean
        public Action transactionalAction(Runnable action) {
            return new Action() {
                @PartnerTransactional
                @Override
                public void perform() {
                    action.run();
                }
            };
        }
    }
}
