package ru.yandex.partner.jsonapi.utils;

import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.crnk.core.engine.information.resource.ResourceField;
import io.crnk.core.engine.information.resource.ResourceFieldType;
import io.crnk.core.engine.information.resource.ResourceInformation;
import io.crnk.core.engine.registry.ResourceRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.lang.Nullable;

import ru.yandex.partner.test.http.json.model.TestHttpResponse;
import ru.yandex.partner.test.utils.TestUtils;

public class UrlUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(UrlUtils.class);
    private static final String ALL_FIELDS_VARIABLE = "$ALL_FIELDS";
    private static final String LAST_CREATED_ID_VARIABLE = "$LAST_CREATED_ID";
    private static final String LAST_CHANGED_ID_VARIABLE = "$LAST_CHANGED_ID";
    private static final Pattern RESOURCE_TYPE_PATTERN = Pattern.compile("^/v1/([^/&?]+)");
    private static final ObjectMapper OBJECT_MAPPER = TestUtils.getObjectMapper();

    private UrlUtils() {
        // Utils
    }

    public static String replaceLastChangeId(TestHttpResponse lastChanged, @Nullable String str) {
        if (str == null || !str.contains(LAST_CHANGED_ID_VARIABLE)) {
            return str;
        }

        if (lastChanged == null) {
            LOGGER.warn("$LAST_CHANGED_ID cannot be used without edit request before current");
            return str;
        }

        try {
            JsonapiResponse jsonapiResponse =
                    OBJECT_MAPPER.readValue(lastChanged.getBody(), JsonapiResponse.class);
            String id = jsonapiResponse.getData().get("id").asText();
            return str.replace(LAST_CHANGED_ID_VARIABLE, id);
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Cannot extract $LAST_CHANGED_ID from:\n" + lastChanged.getBody(), e);
        }

    }

    public static String replaceLastCreatedId(TestHttpResponse lastCreated, @Nullable String str) {
        if (str == null || !str.contains(LAST_CREATED_ID_VARIABLE)) {
            return str;
        }

        if (lastCreated == null) {
            LOGGER.warn("$LAST_CREATED_ID cannot be used without add request before current");
            return str;
        }

        try {
            JsonapiResponse jsonapiResponse =
                    OBJECT_MAPPER.readValue(lastCreated.getBody(), JsonapiResponse.class);
            String id = jsonapiResponse.getData().get("id").asText();
            return str.replace(LAST_CREATED_ID_VARIABLE, id);
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Cannot extract $LAST_CREATED_ID from:\n" + lastCreated.getBody(), e);
        }

    }

    public static String replaceAllFields(ResourceRegistry resourceRegistry, String url) {
        if (!url.contains(ALL_FIELDS_VARIABLE)) {
            return url;
        }

        String resourceType = parseResourceTypeFromUrl(url);

        List<ResourceField> resourceFields = findResourceFieldsByType(resourceRegistry, resourceType);

        return url.replaceAll(
                allFieldsVariableRegexForResource(resourceType),
                allFieldsQueryArgString(resourceType, resourceFields)
        );

    }

    private static String parseResourceTypeFromUrl(String url) {
        Matcher resourceTypeMatcher = RESOURCE_TYPE_PATTERN.matcher(url);

        if (resourceTypeMatcher.find()) {
            return resourceTypeMatcher.group(1);
        } else {
            throw new IllegalStateException("Unable to parse resource type from url: " + url);
        }
    }

    private static List<ResourceField> findResourceFieldsByType(ResourceRegistry resourceRegistry,
                                                                String resourceType) {
        ResourceInformation info = resourceRegistry.getBaseResourceInformation(resourceType);

        if (info != null) {
            return info.getFields();
        } else {
            throw new IllegalStateException("Unable to find resource: " + resourceType);
        }
    }

    private static String allFieldsVariableRegexForResource(String resourceType) {
        return String.format("\\Q%s[%s]\\E", ALL_FIELDS_VARIABLE, resourceType);
    }

    private static String allFieldsQueryArgString(String resourceType, List<ResourceField> resourceFields) {
        String allFieldsString = resourceFields
                .stream()
                .filter(field -> !field.getResourceFieldType().equals(ResourceFieldType.RELATIONSHIP))
                .map(ResourceField::getJsonName)
                .filter(name -> !"linksInformation".equals(name))
                .sorted()
                .collect(Collectors.joining(","));

        return String.format("fields[%s]=%s", resourceType, allFieldsString);
    }

    public static String replaceFields(Map<String, List<String>> fields, String url) {
        var pattern = Pattern.compile("\\$FIELDS\\[([a-zA-Z_0-9]+)]");

        return pattern.matcher(url).replaceAll(match -> {
            var model = match.group(1);
            return "fields[" + model + "]=" + String.join(",", fields.getOrDefault(model, List.of()));
        });
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class JsonapiResponse {
        private Map<String, JsonNode> data;

        public Map<String, JsonNode> getData() {
            return data;
        }

        public void setData(Map<String, JsonNode> data) {
            this.data = data;
        }
    }
}
