package ru.yandex.partner.jsonapi.validation;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import com.fasterxml.jackson.core.util.DefaultIndenter;
import com.fasterxml.jackson.core.util.DefaultPrettyPrinter;
import com.fasterxml.jackson.databind.ObjectMapper;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DynamicContainer;
import org.junit.jupiter.api.DynamicNode;
import org.junit.jupiter.api.DynamicTest;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ArgumentsSource;
import org.mockito.Mockito;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.context.support.MessageSourceAccessor;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.validation.presentation.DefectPresentationRegistry;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.entity.block.type.tags.TagService;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsService;
import ru.yandex.partner.jsonapi.crnk.fields.IncomingApiFields;
import ru.yandex.partner.libs.i18n.TranslatableError;
import ru.yandex.partner.test.http.json.exceptions.JsonUpdateException;
import ru.yandex.partner.test.http.json.model.TestHttpCase;
import ru.yandex.partner.test.http.json.utils.TestCaseManager;
import ru.yandex.partner.test.utils.TestUtils;

import static org.assertj.core.api.Assertions.assertThat;
import static ru.yandex.partner.jsonapi.utils.ApiUtils.generateModelChanges;
import static ru.yandex.partner.jsonapi.utils.ApiUtils.parseJsonMap;

public abstract class AbstractValidationTest<B extends ModelWithId, M extends B> {
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractValidationTest.class);
    private static final DefaultPrettyPrinter PRINTER = defaultPrettyPrinter();
    private static final ObjectMapper OBJECT_MAPPER = TestUtils.getObjectMapper();
    private static boolean needSelfUpdate;

    @Autowired
    ApiFieldsService<M> apiFieldsService;

    @Autowired
    DefectPresentationRegistry<TranslatableError> defectPresentationRegistry;

    @Autowired
    MessageSource messageSource;

    @Autowired
    DSLContext dslContext;

    @Autowired
    TagService tagService;

    MessageSourceAccessor messages;

    @BeforeAll
    static void beforeAll() {
        if (TestCaseManager.needGitDownload()) {
            TestCaseManager.gitDownload(
                    ValidationFileArgumentProvider.getDefaultSourcePath(),
                    ValidationFileArgumentProvider.getGitDirPath()
            );
            Assertions.fail("File(s) downloaded from git");
        }

        needSelfUpdate = TestCaseManager.needSelfUpdate();
    }

    private static DefaultPrettyPrinter defaultPrettyPrinter() {
        DefaultPrettyPrinter.Indenter indenter = new DefaultIndenter();
        DefaultPrettyPrinter printer = new DefaultPrettyPrinter();
        printer.indentObjectsWith(indenter);
        printer.indentArraysWith(indenter);
        return printer;
    }

    @BeforeEach
    void setUp() throws IOException {
        // Преобразуем дефекты валидации из явы в пары "путь" - "ошибка"
        messages = new MessageSourceAccessor(messageSource);
    }

    /**
     * Better ide support
     */
    //@TestFactory
    public Stream<DynamicNode> validationTests() throws IOException, URISyntaxException {
        return new ValidationFileArgumentProvider().provideArguments(null)
                .map(args -> (String) args.get()[0])
                .map(relativeTestPath -> {
                    URL testResource = TestHttpCase.class.getResource(relativeTestPath);
                    String absoluteTestPath = TestUtils.getAbsolutePath(relativeTestPath);
                    try {
                        return DynamicContainer.dynamicContainer(
                                relativeTestPath,
                                new File(absoluteTestPath).toURI(),
                                streamSubtests(relativeTestPath, testResource)
                        );
                    } catch (URISyntaxException e) {
                        throw new RuntimeException(e);
                    }
                });
    }

    protected Stream<? extends DynamicNode> streamSubtests(String relativeTestPath, URL testResource)
            throws URISyntaxException {
        ValidationTestEntity[] entities;
        try (var testsContent = testResource.openStream()) {
            entities = OBJECT_MAPPER.readValue(
                    testsContent,
                    ValidationTestEntity[].class
            );
        } catch (Exception e) {
            throw new RuntimeException("Could not load resource: " + testResource, e);
        }

        String absoluteTestPath = TestUtils.getAbsolutePath(relativeTestPath);

        if (needSelfUpdate) {
            return Stream.of(DynamicTest.dynamicTest(
                    "updating: " + relativeTestPath,
                    new File(absoluteTestPath).toURI(),
                    () -> updateValidationResults(
                            relativeTestPath,
                            entities,
                            validateTestEntityAndGetResults(entities)
                    )
            ));
        }

        return IntStream.range(0, entities.length)
                .mapToObj(index -> {
                    var entity = entities[index];

                    URI resourceUri = UriComponentsBuilder.fromUri(
                            new File(absoluteTestPath).toURI()
                    )
                            .fragment("/" + index)
                            .build().toUri();

                    return DynamicTest.dynamicTest(entity.getName(), resourceUri, () -> {
                        Assumptions.assumeFalse(entity.getSkip(), "test skipped");

                        compareValidationResults(
                                entity,
                                validateEntity(entity)
                        );
                    });
                });
    }

    // Fallback на случай, если динамические тесты начнут некорректно работать,
    //  либо начнут ограничивать возможности
    @ParameterizedTest
    @ArgumentsSource(ValidationFileArgumentProvider.class)
    void validationResultTest(String resourcePath) throws IOException {
        // Читаем ресурс и разбираем в массив тестов
        ValidationTestEntity[] entities = OBJECT_MAPPER.readValue(
                this.getClass().getResourceAsStream(resourcePath),
                ValidationTestEntity[].class
        );

        List<List<DefectInfo<Defect>>> validationResults =
                validateTestEntityAndGetResults(entities);

        if (needSelfUpdate) {
            updateValidationResults(resourcePath, entities, validationResults);
        } else {
            compareValidationResults(entities, validationResults);
        }
    }

    public abstract Map<String, String> getModelPropertyStringMap();

    public abstract Class<B> getBaseClass();

    public abstract B getModelFromDb(Long id);

    public abstract ValidationResult<List<B>, Defect> validateModel(B model, List<ModelChanges<B>> modelChangesList);

    private void updateValidationResults(
            String resourcePath,
            ValidationTestEntity[] entities,
            List<List<DefectInfo<Defect>>> validationResults
    ) {
        LOGGER.info("Self updating file {}", resourcePath);
        File file = TestUtils.prepareFile(TestUtils.getAbsolutePath(resourcePath));

        for (int i = 0; i < entities.length; i++) {
            ValidationTestEntity entity = entities[i];

            if (entity.getSkip()) {
                continue;
            }

            ValidationTestResults result = entity.getResult();
            List<DefectInfo<Defect>> defectInfos = validationResults.get(i);

            if (defectInfos.isEmpty()) {
                result.setOk(true);
                result.setData(new TreeSet<>());
            } else {
                Map<String, String> javaValidationResultMap = translateDefectInfos(defectInfos);
                result.setOk(false);
                Set<ValidationTestResult> collect = EntryStream.of(javaValidationResultMap)
                        .map(e -> new ValidationTestResult(List.of(e.getKey()), List.of(e.getValue())))
                        .toSet();
                result.setData(new TreeSet<>(collect));
            }
        }

        try {
            OBJECT_MAPPER.writer(PRINTER).writeValue(file, entities);
        } catch (IOException e) {
            throw new JsonUpdateException("Error writing json to file", e);
        }
    }

    private void compareValidationResults(
            ValidationTestEntity[] entities,
            List<List<DefectInfo<Defect>>> validationResults
    ) {
        for (int i = 0; i < entities.length; i++) {
            ValidationTestEntity entity = entities[i];

            if (entity.getSkip()) {
                continue;
            }

            // Получаем список с ошибками валидации
            List<DefectInfo<Defect>> defectInfos = validationResults.get(i);

            compareValidationResults(entity, defectInfos);
        }
    }

    private void compareValidationResults(ValidationTestEntity entity, List<DefectInfo<Defect>> defectInfos) {
        ValidationTestResults result = entity.getResult();
        if (result.isOk()) {
            // Если в тесте результат ОК, то в ява валидации не должно быть дефектов
            assertThat(defectInfos).isEmpty();
        } else {
            // Если тест с ошибками, в ява валидации должны быть дефекты
            assertThat(defectInfos).isNotEmpty();

            Map<String, String> javaValidationResultMap = translateDefectInfos(defectInfos);

            TreeSet<ValidationTestResult> resultData = result.getData();
            assertThat(javaValidationResultMap).hasSize(resultData.size());

            // Сравниваем ошибки валидации perl и java
            for (ValidationTestResult testResult : resultData) {
                String path = convertPathToString(testResult.getPath());
                String msg = testResult.getMsgs().get(0);

                assertThat(javaValidationResultMap).containsKey(path);
                assertThat(javaValidationResultMap.get(path)).isEqualTo(msg);
            }
        }
    }

    private Map<String, String> translateDefectInfos(List<DefectInfo<Defect>> defectInfos) {
        Map<String, String> javaValidationResultMap = new HashMap<>();

        for (DefectInfo<Defect> defectInfo : defectInfos) {
            String message = messages.getMessage(
                    defectPresentationRegistry.getPresentation(defectInfo).getMessage());

            String path = convertPathToString(defectInfo.getPath().getNodes().stream()
                    .map(node -> node.toString()).collect(Collectors.toList()));

            javaValidationResultMap.put(path, message);
        }

        return javaValidationResultMap;
    }

    private String convertPathToString(List<String> path) {
        return path.stream().map(part -> getModelPropertyStringMap().getOrDefault(part, part))
                .collect(Collectors.joining("/"));
    }

    private List<List<DefectInfo<Defect>>> validateTestEntityAndGetResults(
            ValidationTestEntity[] entities
    ) throws IOException {
        List<List<DefectInfo<Defect>>> validationResults = new ArrayList<>();

        for (ValidationTestEntity testEntity : entities) {
            validationResults.add(validateEntity(testEntity));
        }

        return validationResults;
    }

    private List<DefectInfo<Defect>> validateEntity(ValidationTestEntity testEntity) throws IOException {
        if (testEntity.getSkip()) {
            return null;
        }

        resetMocks(testEntity);

        // Получаем Map полей модели
        Map<String, ApiField<M>> apiFieldMap =
                StreamEx.of(apiFieldsService.getApiModel().getFields())
                        .toMap(
                                ApiField::getJsonName,
                                apiField -> apiField
                        );

        var model = apiFieldsService.getApiModel().createNewInstance();

        var updatedApiFields = new IncomingApiFields<M>();

        var errorParsing = new LinkedList<DefectInfo<Defect>>();

        // Разбираем patch из теста в Map,
        parseJsonMap(
                model,
                testEntity.getPatch(),
                apiFieldMap,
                updatedApiFields,
                errorParsing
        );

        if (!errorParsing.isEmpty()) {
            return errorParsing;
        }

        Long id = testEntity.getPk().getUniqueId();

        model.setId(id);

        // Читаем блок из базы
        B baseModelFromDb = getModelFromDb(id);
        Class<M> modelClass = apiFieldsService.getApiModel().getModelClass();
        M modelFromDb = modelClass.cast(baseModelFromDb);

        // Генерируем ModelChanges для прочитанного блока
        ModelChanges<M> modelChanges = generateModelChanges(
                modelFromDb,
                model,
                updatedApiFields, apiFieldsService.getApiModel(), apiFieldsService);

        List<ModelChanges<B>> modelChangesList =
                List.of(modelChanges).stream()
                        .map(mc -> mc.castModel(getBaseClass()))
                        .collect(Collectors.toList());

        ValidationResult<List<B>, Defect> modelsListValidationResult = validateModel(
                baseModelFromDb, modelChangesList);

        List<DefectInfo<Defect>> flattenedErrors = modelsListValidationResult
                .getSubResults().get(new PathNode.Index(0))
                .flattenErrors(apiFieldsService.getApiModel().getPathNodeConverterProvider());

        return flattenedErrors;
    }

    private void resetMocks(ValidationTestEntity testEntity) {
        var mockYtPartner = testEntity.getOption("mock_yt_partner", Boolean.class);
        mockYtPartner(Boolean.TRUE.equals(mockYtPartner));
    }

    private void mockYtPartner(boolean mockYtPartner) {
        if (mockYtPartner) {
            Mockito.doReturn(Set.of(1L, 2L, 3L, 4L, 5L, 6L)).when(tagService).getTagIds();
        } else {
            Mockito.reset(tagService);
        }
    }
}
