package ru.yandex.partner.logbrokerapi.configuration;

import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.handler.StatisticsHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.web.embedded.jetty.JettyServerCustomizer;
import org.springframework.boot.web.embedded.jetty.JettyServletWebServerFactory;
import org.springframework.boot.web.server.WebServerFactoryCustomizer;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextClosedEvent;

@Configuration
public class GracefulShutdown {

    //TODO Разделить default configuration на несколько, чтобы можно было без БД положить в ya.make

    @Bean
    public JettyGracefulShutdown jettyGracefulShutdown(@Value("${jetty.server.stop-timeout:120000}") long stopTimeout) {
        return new JettyGracefulShutdown(stopTimeout);
    }

    @Bean
    public WebServerFactoryCustomizer<JettyServletWebServerFactory>
    jettyCustomizer(JettyGracefulShutdown serverCustomizer) {
        return factory -> factory.addServerCustomizers(serverCustomizer);
    }

    public class JettyGracefulShutdown implements ApplicationListener<ContextClosedEvent>, JettyServerCustomizer {
        private long stopTimeout;

        private volatile Server server;

        private final Logger log = LoggerFactory.getLogger(JettyGracefulShutdown.class);

        public JettyGracefulShutdown(long stopTimeout) {
            this.stopTimeout = stopTimeout;
        }

        @Override
        public void onApplicationEvent(ContextClosedEvent event) {
            if (server == null) {
                log.error("Jetty server variable is null, this should not happen!");
                return;
            }
            log.info("Entering shutdown for Jetty.");
            if (!(server.getHandler() instanceof StatisticsHandler)) {
                log.error("Root handler is not StatisticsHandler, graceful shutdown may not work at all!");
            } else {
                log.info("Active requests: " + ((StatisticsHandler) server.getHandler()).getRequestsActive());
            }
            try {
                long begin = System.currentTimeMillis();
                server.stop();
                log.info("Shutdown took " + (System.currentTimeMillis() - begin) + " ms.");
            } catch (Exception e) {
                log.error("Fail to shutdown gracefully.", e);
            }
        }

        @Override
        public void customize(Server server) {
            server.setStopTimeout(stopTimeout);
            server.setStopAtShutdown(true);

            /*
             * StatisticsHandler has to be added for graceful shutdown to work (see
             * https://github.com/eclipse/jetty.project/issues/1549#issuecomment-301102535)
             */
            StatisticsHandler statisticsHandler = new StatisticsHandler();
            statisticsHandler.setHandler(server.getHandler());
            server.setHandler(statisticsHandler);

            this.server = server;
        }
    }
}
