package ru.yandex.partner.runner.service;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import javax.annotation.PostConstruct;

import org.jetbrains.annotations.NotNull;
import org.json.JSONException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.CommandLineRunner;

import ru.yandex.partner.libs.bs.json.BkDataConverter;

public abstract class BkDataCheckRunner<C> implements CommandLineRunner {
    private static final Logger LOGGER = LoggerFactory.getLogger(BkDataCheckRunner.class);
    private static final String UNRECOGNIZED_FINGERPRINT = "Unrecognized field";

    private ExecutorService executor;
    @Value("${threadCount:8}")
    private Integer threadCount;
    @Value("${chunkSize:1000}")
    protected Integer chunkSize;
    @Value("${startFrom:-1}")
    protected long startFrom;

    protected final BkDataConverter bkDataConverter = new BkDataConverter();
    private final AtomicInteger successCount = new AtomicInteger(0);
    private final AtomicInteger count = new AtomicInteger(0);
    private final Map<String, List<Long>> unrecognizedFields = new ConcurrentHashMap<>();

    protected final ReentrantReadWriteLock nextChunkLock = new ReentrantReadWriteLock();

    @PostConstruct
    void init() {
        executor = Executors.newFixedThreadPool(threadCount);
    }

    @Override
    public void run(String... args) throws Exception {
        var tasks = new ArrayList<Future<?>>();
        for (int i = 0; i < threadCount; i++) {
            tasks.add(executor.submit(() -> {
                while (!Thread.currentThread().isInterrupted()) {
                    Map<Long, String> bkDatas = fetchBkDataAndMoveStartFrom();

                    // completed!
                    if (bkDatas.isEmpty()) {
                        return;
                    }

                    var succ = successCount.addAndGet(checkBkDataConversion(bkDatas));
                    var cnt = count.addAndGet(bkDatas.size());
                    LOGGER.warn("Succeeded: {} out of {}; Unrecognized field: {}", succ, cnt, unrecognizedFields);
                }
            }));
        }

        tasks.forEach(task -> {
            try {
                task.get();
            } catch (InterruptedException | ExecutionException e) {
                e.printStackTrace();
            }
        });

        LOGGER.warn("Complete! Succeeded: {} out of {}; Unrecognized field: {}",
                successCount, count, unrecognizedFields);
    }

    @NotNull
    protected Map<Long, String> fetchBkDataAndMoveStartFrom() {
        nextChunkLock.writeLock().lock();
        var bkDatas = fetchBkData(startFrom);

        // completed!
        if (bkDatas.isEmpty()) {
            startFrom = Long.MAX_VALUE;
        } else {
            startFrom = bkDatas.keySet().stream()
                    .mapToLong(it -> it).max()
                    // impossibru
                    .orElse(-1);
            LOGGER.warn("Next chunk starts with {}", startFrom);
        }

        nextChunkLock.writeLock().unlock();
        return bkDatas;
    }

    @NotNull
    protected abstract Map<Long, String> fetchBkData(long start);

    private int checkBkDataConversion(Map<Long, String> bkDatas) {
        C container = prepareContainer(bkDatas);

        return bkDatas.entrySet().stream().parallel().mapToInt(bkDataWithId -> {
            try {
                var bkData = bkDataWithId.getValue();
                if (bkData != null) {
                    checkBkData(bkData, container);
                }
                return 1;
            } catch (AssertionError | IOException | RuntimeException | JSONException e) {
                if (e instanceof IllegalArgumentException
                        && e.getMessage().contains(UNRECOGNIZED_FINGERPRINT)) {
                    var unrecognizedFieldInfo = e.getMessage().substring(UNRECOGNIZED_FINGERPRINT.length());

                    unrecognizedFields.compute(unrecognizedFieldInfo,
                            (k, x) -> {
                                if (x == null) {
                                    x = new ArrayList<>();
                                }
                                x.add(bkDataWithId.getKey());
                                return x;
                            });
                } else {
                    LOGGER.error("Could not perform check for id {}", bkDataWithId.getKey(), e);
                }
            }
            return 0;
        }).sum();
    }

    protected C prepareContainer(Map<Long, String> bkDatas) {
        return null;
    }

    protected abstract void checkBkData(String bkData, C container) throws IOException, AssertionError, JSONException;
}
