package ru.yandex.partner.runner.service;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.Options;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.CommandLineRunner;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.service.statevalidation.block.BlockStateValidationService;

@Profile("validation")
@Component
public class BlockValidationRunner implements CommandLineRunner {
    private static final String FILE_PATH_PARAM = "filePath";
    private static final String THREADS_COUNT_PARAM = "split";
    private static final String PAGE_IDS_PARAM = "pageIds";
    private static final String MODELS_PARAM = "models";

    private static final Logger LOGGER = LoggerFactory.getLogger(BlockValidationRunner.class);
    private static List<BlockStateValidationService<?, ?>> blockStateValidationServices;
    private static int threadsCount;

    @Autowired
    public void setBlockStateValidationService(List<BlockStateValidationService<?, ?>> blockStateValidationServices,
                                               @Value("${cron.validation.threadsCount}") int threadsCount
    ) {
        BlockValidationRunner.blockStateValidationServices = blockStateValidationServices;
        BlockValidationRunner.threadsCount = threadsCount;
    }

    @Override
    public void run(String... args) {
        var argList = Arrays.asList(args);
        //Используется логирования уровня WARN, так как этот метод используется в oneshot, где стоит такой уровень
        //логирования, чтобы не заспамить логами
        LOGGER.warn("Block manual validation started with params: {}", argList);
        if (args.length < 1) {
            LOGGER.warn("#END");
            System.exit(0);
        }

        try {
            CommandLineParser parser = new DefaultParser();
            CommandLine cmd = parser.parse(getOptions(), args, false);

            List<Long> ids;
            if (cmd.hasOption(FILE_PATH_PARAM)) {
                Path path = Paths.get(cmd.getOptionValue(FILE_PATH_PARAM));
                var data = Files.lines(path).collect(Collectors.joining());
                LOGGER.warn(data);
                ids = parseToPageIds(data);
            } else if (cmd.hasOption(PAGE_IDS_PARAM)) {
                ids = parseToPageIds(cmd.getOptionValue(PAGE_IDS_PARAM));
            } else {
                ids = new ArrayList<>();
            }

            int curThreadCount;
            if (cmd.hasOption(THREADS_COUNT_PARAM)) {
                curThreadCount = Math.min(Math.min(ids.size(), threadsCount),
                        Integer.parseInt(cmd.getOptionValue(THREADS_COUNT_PARAM)));
                if (curThreadCount < 1) {
                    throw new IllegalArgumentException("Wrong arguments");
                }
            } else {
                curThreadCount = 1;
            }

            Set<String> modelsToValidate;
            if (cmd.hasOption(MODELS_PARAM)) {
                modelsToValidate = parseUniqueStrings(cmd.getOptionValue(MODELS_PARAM));
            } else {
                // по дефолту валидируем все модели
                modelsToValidate = blockStateValidationServices.stream()
                        .map(BlockStateValidationService::getValidatedModelName)
                        .collect(Collectors.toSet());
            }

            validate(ids, curThreadCount, modelsToValidate);
            LOGGER.warn("#END");
            System.exit(0);
        } catch (Exception e) {
            LOGGER.error("Validation interrupted by exception: {}", e.getMessage(), e);
            LOGGER.warn("#END");
            System.exit(1);
        }

    }

    private void validate(
            List<Long> ids, int curThreadCount, Set<String> models) throws ExecutionException, InterruptedException {
        LOGGER.warn("Going to validate models: {}", models);
        LOGGER.warn("Validation with page ids: {}", ids);
        if (!ids.isEmpty()) {
            if (curThreadCount == 1) {
                blockStateValidationServices.stream()
                        .filter(s -> models.contains(s.getValidatedModelName()))
                        .forEach(s -> s.validateByPageIds(ids));
            } else {
                Executor executor = Executors.newFixedThreadPool(curThreadCount);
                List<CompletableFuture<Void>> threads =
                        new ArrayList<>(curThreadCount * blockStateValidationServices.size());

                blockStateValidationServices.stream()
                        .filter(s -> models.contains(s.getValidatedModelName()))
                        .forEach(service -> {
                            for (int i = 0; i < curThreadCount; i++) {
                                int part = i;
                                List<Long> idsToValidate =
                                        ids.stream().filter(it -> it % curThreadCount == part).toList();
                                CompletableFuture<Void> completableFuture = CompletableFuture
                                        .runAsync(() -> service.validateByPageIds(idsToValidate), executor);
                                threads.add(completableFuture);
                            }
                        });
                CompletableFuture<Void> combinedFuture =
                        CompletableFuture.allOf(threads.toArray(new CompletableFuture[]{}));
                combinedFuture.get();
            }
        }
        LOGGER.warn("Validation finished");
    }

    private List<Long> parseToPageIds(String ids) {
        return parseUniqueStrings(ids).stream()
                .map(Long::parseLong)
                .collect(Collectors.toList());
    }

    private Set<String> parseUniqueStrings(String param) {
        return Arrays.stream(param.split("\\s*,\\s*|\\s+"))
                .collect(Collectors.toSet());
    }

    private Options getOptions() {
        Options options = new Options();
        options.addOption("f", FILE_PATH_PARAM, true, "file path");
        options.addOption("s", THREADS_COUNT_PARAM, true, "split in threads");
        options.addOption("p", PAGE_IDS_PARAM, true, "page ids");
        options.addOption("m", MODELS_PARAM, true, "validation models name");
        return options;
    }
}
