package ru.yandex.partner.runner.service;

import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import NPartner.Page;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.impl.DSL;
import org.json.JSONException;
import org.junit.jupiter.api.Assertions;
import org.skyscreamer.jsonassert.JSONAssert;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.bs.BkDataException;
import ru.yandex.partner.core.bs.BkDataRepository;
import ru.yandex.partner.core.bs.enrich.BkDataService;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.libs.bs.json.PageBkDataComparator;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static org.jooq.impl.DSL.field;
import static org.jooq.impl.DSL.max;
import static org.jooq.impl.DSL.select;

@Profile("page-bk-data-enrich")
@Component
public class PageBkDataEnrichCheckRunner extends
        BkDataCheckRunner<Map<Long, List<Page.TPartnerPage.TBlock>>> {
    private static final Table<Record> BK_EDIT_PAGE_202X = DSL.table("partner_logs.bk_edit_page_2022");

    @Autowired
    private DSLContext dslContext;
    @Autowired
    private BkDataRepository<BaseBlock, Page.TPartnerPage.TBlock> bkDataRepository;
    @Autowired
    private BkDataService bkDataService;

    private PageBkDataComparator comparator = new PageBkDataComparator();
    private ObjectMapper om = new ObjectMapper();

    @NotNull
    @Override
    protected Map<Long, String> fetchBkData(long start) {
        var pageId = field("page_id", Long.class);
        var request = field("request", String.class);
        var dt = field("dt", Date.class);
        return dslContext.select(pageId, DSL.function("uncompress", String.class, request).as("request"))
                .from(BK_EDIT_PAGE_202X.as("bk"))
                .where(pageId.greaterThan(start)
                        .and(dt.eq(select(max(dt)).from(BK_EDIT_PAGE_202X.as("tmp_bk"))
                                .where(field("tmp_bk.page_id", Long.class).eq(field("bk.page_id", Long.class)))))
                )
                .orderBy(pageId)
                .limit(chunkSize)
                .fetchMap(pageId, request);
    }

    @Override
    protected Map<Long, List<Page.TPartnerPage.TBlock>> prepareContainer(Map<Long, String> bkDatas) {
        return bkDataService.getSupportedBlocksForPages(bkDatas.keySet(), Set.of(BaseBlock.ID, BaseBlock.PAGE_ID))
                .stream()
                .collect(groupingBy(
                        BaseBlock::getPageId,
                        collectingAndThen(
                                mapping(BaseBlock::getId, toList()),
                                ids -> {
                                    try {
                                        return bkDataRepository.getBkData(ids, true);
                                    } catch (BkDataException ex) {
                                        throw new RuntimeException("Could not collect block bkdatas", ex);
                                    }
                                }
                        )
                ));
    }

    @Override
    protected void checkBkData(String bkData, Map<Long, List<Page.TPartnerPage.TBlock>> container)
            throws IOException, JSONException {
        // bk data is weird here
        bkData = om.readTree(bkData).iterator().next().iterator().next().toString();
        var parseResult = bkDataConverter.convertPageJsonToProto(bkData);
        Page.TPartnerPage tPartnerPage = parseResult.getMessage();

        Assertions.assertTrue(parseResult.getErrors().isEmpty(),
                "Should not have parse errors: " + parseResult.getErrors());

        var blocksGeneratedBkData = container.get(tPartnerPage.getPageID());

        var pageBuilder = Page.TPartnerPage.newBuilder(tPartnerPage)
                .clearRtbVideo();

        bkDataService.enrichPageWithBlocks(pageBuilder, blocksGeneratedBkData);

        var dataAfterConvert = bkDataConverter.convertProtoToJson(pageBuilder.build());
        JSONAssert.assertEquals(bkData, dataAfterConvert, comparator);
    }
}
