package ru.yandex.partner.runner.service;

import java.io.IOException;
import java.util.Date;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.impl.DSL;
import org.json.JSONException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.block.service.OperationMode;
import ru.yandex.partner.core.entity.dsp.model.Dsp;
import ru.yandex.partner.core.entity.dsp.repository.DspTypedRepository;
import ru.yandex.partner.core.entity.page.type.bkdata.PageBkDataValidatorProvider;

import static org.jooq.impl.DSL.field;
import static org.jooq.impl.DSL.max;
import static org.jooq.impl.DSL.select;
import static ru.yandex.partner.core.entity.block.type.dsps.BlockWithDspsRepositoryTypeSupport.DSP_PROPS_FOR_BLOCK;

@Profile("page-bk-data-validation")
@Component
public class PageBkDataValidationRunner extends BkDataCheckRunner<Map<Long, Dsp>> {
    private static final Table<Record> BK_EDIT_PAGE_202X = DSL.table("partner_logs.bk_edit_page_2022");

    @Autowired
    private DSLContext dslContext;
    @Autowired
    private DspTypedRepository dspTypedRepository;
    @Autowired
    private PageBkDataValidatorProvider pageBkDataValidatorProvider;
    private ObjectMapper om = new ObjectMapper();

    @Override
    protected Map<Long, Dsp> prepareContainer(Map<Long, String> bkDatas) {
        return dspTypedRepository.getAll(QueryOpts.forClass(Dsp.class).withProps(DSP_PROPS_FOR_BLOCK)).stream()
                .map(it -> (Dsp) it)
                .collect(Collectors.toMap(Dsp::getId, Function.identity()));
    }

    @NotNull
    @Override
    protected Map<Long, String> fetchBkData(long start) {
        var pageId = field("page_id", Long.class);
        var request = field("request", String.class);
        var dt = field("dt", Date.class);
        return dslContext.select(pageId, DSL.function("uncompress", String.class, request).as("request"))
                .from(BK_EDIT_PAGE_202X.as("bk"))
                .where(pageId.greaterThan(start)
                        .and(dt.eq(select(max(dt)).from(BK_EDIT_PAGE_202X.as("tmp_bk"))
                                .where(field("tmp_bk.page_id", Long.class).eq(field("bk.page_id", Long.class)))))
                )
                .orderBy(pageId)
                .limit(chunkSize)
                .fetchMap(pageId, request);
    }

    @Override
    protected void checkBkData(String bkData, Map<Long, Dsp> dspContainer) throws IOException, JSONException {
        // bk data is weird here
        bkData = om.readTree(bkData).iterator().next().iterator().next().toString();
        var pageParseResult = bkDataConverter.convertPageJsonToProto(bkData);
        if (!pageParseResult.getErrors().isEmpty()) {
            throw new RuntimeException("Page parsed with errors: " + pageParseResult.getErrors());
        }
        var validationResult = pageBkDataValidatorProvider.validator(OperationMode.CRON, dspContainer)
                .apply(pageParseResult.getMessage());

        if (validationResult.hasAnyErrors()) {
            throw new RuntimeException("Page validated with errors: " + validationResult.flattenErrors());
        }
    }
}
