package ru.yandex.partner.testapi;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.util.MultiValueMap;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.server.ResponseStatusException;

import ru.yandex.partner.testapi.exceptions.TestApiConcurrentException;
import ru.yandex.partner.testapi.fixture.FixtureResult;
import ru.yandex.partner.testapi.service.testcase.FixtureListDto;
import ru.yandex.partner.testapi.service.testcase.TestApiService;

/**
 * Класс контроллера для наливки БД для автотество
 */
@RestController
public class TestApiController {
    private static final Logger LOGGER = LoggerFactory.getLogger(TestApiController.class);
    private final ExecutorService executor = Executors.newSingleThreadExecutor();
    private final TestApiService testApiService;
    private Future<?> result;

    @Autowired
    public TestApiController(TestApiService testApiService) {
        this.testApiService = testApiService;
    }

    /**
     * Метод запускает перенаполнение базы данных
     * Базовый URL для маппинга /testapi/start/{testcase}
     *
     * @param testcase имя теста
     */
    @PostMapping(path = {"/testapi/start/{testcase}"})
    public void justStart(
            @PathVariable(value = "testcase") String testcase
    ) {
        LOGGER.info("Start testcase {}", testcase);
        result = executor.submit(() -> testApiService.processResourceTestcase(testcase));
    }

    /**
     * use {@link #justStart(String)}
     */
    @GetMapping(path = {"/testapi/start/{testcase}"})
    @Deprecated
    public void justStartDeprecated(
            @PathVariable(value = "testcase") String testcase
    ) {
        LOGGER.info("GET request /testapi/start/{}", testcase);
        justStart(testcase);
    }

    /**
     * Метод запускает перенаполнение базы данных
     * Базовый URL для маппинга /testapi/start/
     *
     * @param fixtureListDto список фикстур
     */
    @PostMapping(path = {"/testapi/start"})
    public void justStart(
            @RequestBody FixtureListDto fixtureListDto
    ) {
        LOGGER.info("POST request /testapi/start/ with FixtureList {}", fixtureListDto);
        result = executor.submit(() -> testApiService.processFixtureList(fixtureListDto));
    }

    /**
     * Метод запускает перенаполнение базы данных и дожидается ответа
     * Базовый URL для маппинга /testapi/wait/{testcase}
     *
     * @param testcase имя теста
     * @return FixtureStateListDto результат выполнения тесткейса
     */
    @PostMapping(path = {"/testapi/wait/{testcase}"})
    public FixtureStateListDto startAndWait(
            @PathVariable(value = "testcase") String testcase
    ) {
        try {
            testApiService.processResourceTestcase(testcase);
            return getFixtureStateListDto(testApiService.getFixtureStateMap());
        } catch (TestApiConcurrentException e) {
            LOGGER.error(e.getMessage(), e);
            throw new ResponseStatusException(HttpStatus.CONFLICT);
        } catch (IllegalArgumentException e) {
            var text = "Testcase not found: " + testcase;
            LOGGER.warn(text, e);
            throw new ResponseStatusException(HttpStatus.NOT_FOUND, text, e);
        } catch (Exception e) {
            LOGGER.error(e.getMessage(), e);
            throw new ResponseStatusException(HttpStatus.INTERNAL_SERVER_ERROR, e.getMessage(), e);
        }
    }

    /**
     * use {@link #startAndWait(String)}
     */
    @GetMapping(path = {"/testapi/wait/{testcase}"})
    @Deprecated
    public FixtureStateListDto startAndWaitDeprecated(
            @PathVariable(value = "testcase") String testcase
    ) {
        LOGGER.info("GET request /testapi/wait/{}", testcase);
        return startAndWait(testcase);
    }

    /**
     * Метод запускает перенаполнение базы данных и дожидается ответа
     * Базовый URL для маппинга /testapi/wait/
     *
     * @param fixtureListDto список фикстур
     * @return FixtureStateListDto результат выполнения тесткейса
     */
    @PostMapping(path = {"/testapi/wait/"})
    public FixtureStateListDto startAndWait(
            @RequestBody FixtureListDto fixtureListDto
    ) {
        LOGGER.info("POST request /testapi/wait/ with FixtureList {}", fixtureListDto);
        try {
            testApiService.processFixtureList(fixtureListDto);
            return getFixtureStateListDto(testApiService.getFixtureStateMap());
        } catch (TestApiConcurrentException e) {
            LOGGER.error(e.getMessage(), e);
            throw new ResponseStatusException(HttpStatus.CONFLICT);
        } catch (Exception e) {
            LOGGER.error(e.getMessage(), e);
            throw new ResponseStatusException(HttpStatus.INTERNAL_SERVER_ERROR, e.getMessage(), e);
        }
    }

    /**
     * Метод проверяет, отработала ли футура,
     * ответственная за наполнение базы
     * Базовый URL для маппинга /testapi/isdone/
     *
     * @return пустая строка
     */
    @GetMapping(path = {"/testapi/isdone"})
    public Boolean isDone() {
        LOGGER.info("GET request /testapi/isdone");
        return result != null && result.isDone();
    }

    @GetMapping(path = "/testapi/fixtures")
    public FixtureStateListDto getFixtures() {
        LOGGER.info("GET request /testapi/fixtures");
        return getFixtureStateListDto(testApiService.getFixtureStateMap());
    }

    public FixtureStateListDto getFixtureStateListDto(MultiValueMap<String, FixtureResult> map) {
        return new FixtureStateListDto(
                map.entrySet().stream()
                        .map(FixtureStateDto::new)
                        .collect(Collectors.toList())
        );
    }

    @SuppressWarnings("unused")
    private static class FixtureStateListDto {
        private List<FixtureStateDto> fixtures;

        FixtureStateListDto(List<FixtureStateDto> fixtures) {
            this.fixtures = fixtures;
        }

        public List<FixtureStateDto> getFixtures() {
            return fixtures;
        }
    }

    @SuppressWarnings("unused")
    private static class FixtureStateDto {
        private String name;
        private List<FixtureResult> results;

        FixtureStateDto(Map.Entry<String, List<FixtureResult>> entry) {
            name = entry.getKey();
            results = entry.getValue();
        }

        public String getName() {
            return name;
        }

        public List<FixtureResult> getResults() {
            return results;
        }
    }
}
