package ru.yandex.partner.testapi.filter;

import java.io.IOException;
import java.io.UnsupportedEncodingException;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.server.ServletServerHttpResponse;
import org.springframework.stereotype.Component;
import org.springframework.web.filter.CommonsRequestLoggingFilter;
import org.springframework.web.util.ContentCachingRequestWrapper;
import org.springframework.web.util.ContentCachingResponseWrapper;
import org.springframework.web.util.WebUtils;

@Component
@ParametersAreNonnullByDefault
public class RequestAndResponseLoggingFilter extends CommonsRequestLoggingFilter {

    @Autowired
    public RequestAndResponseLoggingFilter() {
        this.setIncludeQueryString(true);
        this.setIncludePayload(true);
        this.setMaxPayloadLength(10000);
        this.setIncludeHeaders(true);
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response,
                                    FilterChain filterChain) throws ServletException, IOException {
        HttpServletRequest requestToUse = request;

        if ((request instanceof ContentCachingRequestWrapper)) {
            requestToUse = new ContentCachingRequestWrapper(request, getMaxPayloadLength());
        }

        if (!(response instanceof ContentCachingResponseWrapper)) {
            response = new ContentCachingResponseWrapper(response);
        }

        logger.info(createMessage(requestToUse, "REQUEST: ", ""));
        try {
            filterChain.doFilter(requestToUse, response);
        } finally {
            logger.info(createMessage(response, "RESPONSE: ", ""));
        }
    }

    @SuppressWarnings("SameParameterValue")
    private String createMessage(HttpServletResponse response, String prefix, String suffix) throws IOException {
        return prefix +
                "code=" + response.getStatus() +
                ";headers=" + new ServletServerHttpResponse(response).getHeaders() +
                ";payload=" + getResponsePayload(response) +
                suffix;
    }

    private String getResponsePayload(HttpServletResponse response) throws IOException {
        ContentCachingResponseWrapper wrapper = WebUtils.getNativeResponse(response,
                ContentCachingResponseWrapper.class);
        if (wrapper != null) {

            byte[] buf = wrapper.getContentAsByteArray();
            if (buf.length > 0) {
                int length = Math.min(buf.length, getMaxPayloadLength());
                try {
                    String body = new String(buf, 0, length, wrapper.getCharacterEncoding());
                    wrapper.copyBodyToResponse();
                    return body;
                } catch (UnsupportedEncodingException ex) {
                    // NOOP
                }
            }
        }
        return "[unknown]";
    }
}
