package ru.yandex.partner.testapi.fixture;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.TableField;

import ru.yandex.partner.testapi.exceptions.TestApiException;

public class FixtureUtils {

    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper()
            .configure(DeserializationFeature.USE_LONG_FOR_INTS, true);


    private FixtureUtils() { }

    public static <R extends Record> Long getNextId(Long currentId,
                                                    DSLContext dslContext,
                                                    TableField<R, Long> idField,
                                                    TableField<R, Long> nextIdField) {
        R record = dslContext
                .selectFrom(idField.getTable())
                .where(idField.eq(currentId))
                .fetchOne();
        if (record != null) {
            Long resultId = record.get(nextIdField);
            dslContext.update(idField.getTable())
                    .set(nextIdField, resultId + 1)
                    .where(idField.eq(currentId))
                    .execute();
            return resultId;
        } else {
            long resultId = 1L;
            record = idField.getTable().newRecord();
            record.set(idField, currentId);
            record.set(nextIdField, resultId + 1);
            dslContext.insertInto(idField.getTable())
                    .set(record)
                    .execute();
            return resultId;
        }
    }

    /**
     * Поддерживает одиночный объект
     * ```
     * "opts" : {
     *     "caption" : "bla"
     * }
     * ```
     * и массив
     * ```
     * "opts" : [
     *      { "caption" : "bla" },
     *      { "caption" : "foo" }
     * ]
     * ```
     * @param json
     * @return
     */
    public static List<Map<String, Object>> extractRecordPatches(JsonNode json) {
        List<Map<String, Object>> result;
        if (json != null && !json.isNull() && !json.isMissingNode()) {
            try {
                if (json.isArray()) {
                    result = OBJECT_MAPPER.readValue(
                            json.toString(),
                            OBJECT_MAPPER.getTypeFactory().constructCollectionType(List.class, Map.class)
                    );
                } else {
                    result = new ArrayList<>();
                    result.add(OBJECT_MAPPER.readValue(
                            json.toString(),
                            OBJECT_MAPPER.getTypeFactory().constructType(Map.class))
                    );
                }
            } catch (IOException e) {
                throw new TestApiException("Error during parse opts", e);
            }
        } else {
            result = new ArrayList<>(1);
            result.add(Collections.emptyMap());
        }

        return result;
    }
}
