package ru.yandex.partner.testapi.fixture.block;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.InsertValuesStep3;
import org.jooq.Record;
import org.jooq.Table;

import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.dbschema.partner.tables.records.BlockDspsRecord;
import ru.yandex.partner.dbschema.partner.tables.records.MediaSizesRecord;
import ru.yandex.partner.testapi.fixture.Fixture;
import ru.yandex.partner.testapi.fixture.FixtureContext;
import ru.yandex.partner.testapi.fixture.FixtureResult;
import ru.yandex.partner.testapi.fixture.FixtureUtils;
import ru.yandex.partner.testapi.fixture.dsp.DspsFixture;
import ru.yandex.partner.testapi.utils.ResourceUtils;

import static ru.yandex.partner.dbschema.partner.Tables.MEDIA_SIZES;

public abstract class AbstractBlockFixture implements Fixture {
    private final DSLContext dslContext;
    protected final Map<String, Field<?>> fieldTypes;

    private ObjectMapper objectMapper;

    public AbstractBlockFixture(DSLContext dslContext) {
        this.dslContext = dslContext;
        this.fieldTypes = StreamEx.of(Arrays.stream(getTable().fields()))
                .toMap(Field::getName, f -> f);
        this.objectMapper = new ObjectMapper();
        this.objectMapper.configure(DeserializationFeature.USE_LONG_FOR_INTS, true);
    }

    public abstract String getPublicIDPrefix();

    public abstract String getPageFixtureName();

    public String getDspFixtureName() {
        return DspsFixture.FIXTURE_NAME;
    }

    public abstract String getPageIdFieldName();

    protected abstract Long getNextBlockIdFromSeq(Long pageId);

    protected abstract void updatePageBlocksCount(Long pageId);

    @Override
    public List<FixtureResult> createAndSave(JsonNode optsJson, FixtureContext fixtureContext) {
        List<Map<String, Object>> optsList = FixtureUtils.extractRecordPatches(optsJson);

        Long pageId = getPageId(fixtureContext);
        List<FixtureResult> resultList = new ArrayList<>();
        for (Map<String, Object> params : optsList) {
            Long blockId = getNextBlockIdFromSeq(pageId);
            resultList.add(createAndSaveForPageIdAndBlockId(pageId, blockId, params));
        }
        return resultList;
    }

    @Override
    public List<String> getFixtureDepends() {
        return List.of(getPageFixtureName(), getDspFixtureName());
    }

    protected FixtureResult createAndSaveForPageIdAndBlockId(Long pageId, Long blockId,
                                                             Map<String, Object> optsMap) {
        getDslContext().insertInto(getTable())
                .set(getBlockRecord(pageId, blockId, optsMap))
                .execute();

        updatePageBlocksCount(pageId);
        updateMediaSizes(pageId, blockId);
        updateBlockDsps(pageId, blockId);

        return new FixtureResult(
                pageId + " " + blockId,
                Map.of(
                        "id", blockId,
                        getPageIdFieldName(), pageId,
                        "public_id", getPublicID(pageId, blockId)
                )
        );
    }

    private void updateBlockDsps(Long pageId, Long blockId) {
        Function<Long, BlockDspsRecord> newDspRecord = dspId -> {
            var dspRecord = new BlockDspsRecord();
            dspRecord.setPageId(pageId);
            dspRecord.setBlockId(blockId);
            dspRecord.setIsDeleted(0L);
            dspRecord.setDspId(dspId);
            return dspRecord;
        };

        List<BlockDspsRecord> blockDspsRecords = getDspIds().stream()
                .map(newDspRecord)
                .toList();

        getDslContext().batchInsert(blockDspsRecords).execute();
    }

    protected Long getPageId(FixtureContext fixtureContext) {
        List<FixtureResult> pageFixtureResults = fixtureContext.valueByFixtureName(getPageFixtureName());
        if (pageFixtureResults == null || pageFixtureResults.isEmpty()) {
            throw new RuntimeException("PageId fixture fail");
        }
        return (Long) pageFixtureResults.get(0).getParams().get("id");
    }

    public String getPublicID(Long pageId, Long blockId) {
        return String.format("%s-%d-%d", getPublicIDPrefix(), pageId, blockId);
    }

    protected DSLContext getDslContext() {
        return dslContext;
    }

    protected String readOptsFromResource() {
        return ResourceUtils.readAsString("/data/fixtures/" + getFixtureName() + "/opts.json");
    }

    protected abstract Table<? extends Record> getTable();

    protected abstract Record getBlockRecord(Long pageId, Long blockId, Map<String, Object> optsMap);

    private void updateMediaSizes(Long pageId, Long blockId) {
        InsertValuesStep3<MediaSizesRecord, Long, Long, String> insert =
                getDslContext().insertInto(MEDIA_SIZES, MEDIA_SIZES.PAGE_ID, MEDIA_SIZES.BLOCK_ID, MEDIA_SIZES.TYPE);
        for (String mediaType : getMediaTypes()) {
            insert = insert.values(pageId, blockId, mediaType);
        }
        insert.execute();
    }

    protected List<String> getMediaTypes() {
        return List.of();
    }

    protected List<Long> getDspIds() {
        return List.of(CoreConstants.DSP_DIRECT_ID);
    }
}
