package ru.yandex.partner.testapi.fixture.mobile;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.databind.JsonNode;
import org.jooq.DSLContext;

import ru.yandex.partner.dbschema.partner.tables.records.MobileAppOwnerRecord;
import ru.yandex.partner.dbschema.partner.tables.records.MobileAppRecord;
import ru.yandex.partner.libs.utils.JooqUtils;
import ru.yandex.partner.testapi.fixture.Fixture;
import ru.yandex.partner.testapi.fixture.FixtureContext;
import ru.yandex.partner.testapi.fixture.FixtureResult;
import ru.yandex.partner.testapi.fixture.FixtureUtils;

import static ru.yandex.partner.dbschema.partner.tables.MobileApp.MOBILE_APP;
import static ru.yandex.partner.dbschema.partner.tables.MobileAppOwner.MOBILE_APP_OWNER;

public abstract class AbstractMobileAppFixture implements Fixture {
    private final DSLContext dslContext;
    private final LocalDateTime localDateTime = LocalDateTime.of(2020, 3, 31, 12, 0);

    public AbstractMobileAppFixture(DSLContext dslContext) {
        this.dslContext = dslContext;
    }

    public abstract String getUserFixtureName();

    public abstract Long getType();

    public abstract String getStoreId();

    public abstract String getStoreUrl();

    public LocalDateTime getLocalDateTime() {
        return localDateTime;
    }

    public Long getUserId(FixtureContext fixtureContext) {
        List<FixtureResult> userFixtureResults = fixtureContext.valueByFixtureName(getUserFixtureName());
        if (userFixtureResults == null || userFixtureResults.isEmpty()) {
            throw new RuntimeException("User fixture fail");
        }
        return (Long) userFixtureResults.get(0).getParams().get("id");
    }

    public MobileAppRecord getDefaultMobileAppRecord(Map<String, Object> patch) {
        MobileAppRecord mobileAppRecord = new MobileAppRecord();
        mobileAppRecord.setType(getType());
        mobileAppRecord.setStoreId(getStoreId());
        mobileAppRecord.setStoreUrl(getStoreUrl());
        mobileAppRecord.setOpts("{\"is_mobile_mediation\": 0}");
        mobileAppRecord.setMultistate(4L); // approved
        mobileAppRecord.setCreateDate(getLocalDateTime());

        JooqUtils.applyPatchToRecord(mobileAppRecord, patch);

        return mobileAppRecord;
    }

    public MobileAppRecord getMobileAppRecord(Map<String, Object> optsMap) {
        return getDefaultMobileAppRecord(optsMap);
    }

    public MobileAppOwnerRecord getDefaultMobileAppOwnerRecord(Long appId, Long userId) {
        MobileAppOwnerRecord mobileAppOwnerRecord = new MobileAppOwnerRecord();
        mobileAppOwnerRecord.setAppId(appId);
        mobileAppOwnerRecord.setUserId(userId);
        return mobileAppOwnerRecord;
    }

    public MobileAppOwnerRecord getMobileAppOwnerRecord(Long appId, Long userId) {
        return getDefaultMobileAppOwnerRecord(appId, userId);
    }

    @Override
    public List<String> getFixtureDepends() {
        return List.of(getUserFixtureName());
    }

    @Override
    public List<FixtureResult> createAndSave(JsonNode optsJson, FixtureContext fixtureContext) {
        List<Map<String, Object>> optsList = FixtureUtils.extractRecordPatches(optsJson);

        Long userId = getUserId(fixtureContext);

        List<FixtureResult> list = new ArrayList<>();
        for (Map<String, Object> opts: optsList) {
            MobileAppRecord mobileAppRecord = getMobileAppRecord(opts);
            MobileAppRecord mobileAppRecordId = dslContext.insertInto(MOBILE_APP)
                    .set(mobileAppRecord)
                    .returning(MOBILE_APP.ID)
                    .fetchOne();
            // Получаем id приложения из автоинкрементного поля
            Long appId = mobileAppRecordId.getId();
            MobileAppOwnerRecord mobileAppOwnerRecord = getMobileAppOwnerRecord(appId, userId);
            dslContext.insertInto(MOBILE_APP_OWNER)
                    .set(mobileAppOwnerRecord).execute();
            list.add(new FixtureResult(
                    appId.toString(),
                    Map.of(
                            "id", appId,
                            "type", mobileAppRecord.getType(),
                            "store_id", mobileAppRecord.getStoreId(),
                            "store_url", mobileAppRecord.getStoreUrl(),
                            "user_id", userId
                    )
            ));
        }
        return list;
    }
}
