package ru.yandex.partner.testapi.fixture.service.tus;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.reactive.function.client.WebClientResponseException;

import ru.yandex.partner.testapi.exceptions.TestApiException;

@Service(TusServiceImpl.BEAN_NAME)
public class TusServiceImpl implements TusService {
    public static final String BEAN_NAME = "TusService";
    private static final Logger LOGGER = LoggerFactory.getLogger(TusService.class);
    private final WebClient webClient;
    private final String tusConsumer;
    private final String oauthToken;
    private final List<TusUser> tusUsersCache;
    private final String tusEnv;
    private int position;

    @Autowired
    public TusServiceImpl(@Value("${tus.url}") String baseUrl,
                          @Value("${tus.consumer}") String tusConsumer,
                          @Value("${tus.oauth}") String oauthToken,
                          @Value("${tus.env}") String tusEnv) {
        this(WebClient.create(baseUrl), tusConsumer, oauthToken, tusEnv);
    }

    TusServiceImpl(WebClient webClient, String tusConsumer, String oauthToken, String tusEnv) {
        this.webClient = webClient;
        this.tusConsumer = tusConsumer;
        this.oauthToken = oauthToken;
        this.tusEnv = tusEnv;
        this.tusUsersCache = new ArrayList<>();
        this.position = 0;
    }

    @Override
    public void clearUserCachePosition() {
        position = 0;
    }

    @Override
    public void clearUserCache() {
        clearUserCachePosition();
        tusUsersCache.clear();
    }

    @Override
    public TusUser getOrCreateUser() {
        TusUser tusUser;
        if (position < tusUsersCache.size()) {
            tusUser = tusUsersCache.get(position);
        } else {
            tusUser = createUser();
            tusUsersCache.add(tusUser);
        }

        position++;
        return tusUser;
    }

    @Override
    public TusUser createUser() {
        LOGGER.info("Call TusService createUser");
        try {
            ResponseEntity<TusDto> responseEntity = webClient.post()
                    .uri("1/create_account/portal/")
                    .header("Authorization", "OAuth " + oauthToken)
                    .body(BodyInserters.fromFormData("env", tusEnv).with("tus_consumer", tusConsumer))
                    .retrieve()
                    .toEntity(TusDto.class)
                    .block();

            LOGGER.info("End http request");

            if (responseEntity == null) {
                LOGGER.error("Error during create user. Response is null");
                throw new TestApiException("Error during create user. Response is null");
            }

            if (!responseEntity.getStatusCode().is2xxSuccessful()) {
                LOGGER.error("Error during create user. Response status = {}",
                        responseEntity.getStatusCode().value());
                throw new TestApiException("Error during create user. Response status = "
                        + responseEntity.getStatusCode().value());
            }

            if (responseEntity.getBody() == null) {
                LOGGER.error("Error during create user. Response body is null");
                throw new TestApiException("Error during create user. Response body is null");
            }

            if (!"ok".equals(responseEntity.getBody().getStatus())) {
                LOGGER.error("Error during create user. Response body status = {}",
                        responseEntity.getBody().getStatus());
                throw new TestApiException("Error during create user. Response body status = "
                        + responseEntity.getBody().getStatus());
            }

            return responseEntity.getBody().getAccount();
        } catch (WebClientResponseException e) {
            LOGGER.error("Error during exchange {}", e.getMessage());
            throw new TestApiException("Error during exchange", e);
        }
    }

    @SuppressWarnings("unused")
    public static class TusDto {
        private TusUser account;
        private String status;

        TusUser getAccount() {
            return account;
        }

        void setAccount(TusUser account) {
            this.account = account;
        }

        String getStatus() {
            return status;
        }

        void setStatus(String status) {
            this.status = status;
        }
    }
}
