package ru.yandex.partner.testapi.fixture.user;

import java.io.IOException;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import org.jooq.DSLContext;
import org.slf4j.Logger;

import ru.yandex.partner.core.entity.user.model.UserOptsHelper;
import ru.yandex.partner.dbschema.partner.tables.records.ContractsRecord;
import ru.yandex.partner.dbschema.partner.tables.records.UserAdfoxRecord;
import ru.yandex.partner.dbschema.partner.tables.records.UserFeaturesRecord;
import ru.yandex.partner.dbschema.partner.tables.records.UserRoleRecord;
import ru.yandex.partner.dbschema.partner.tables.records.UsersRecord;
import ru.yandex.partner.libs.utils.JooqUtils;
import ru.yandex.partner.testapi.exceptions.TestApiException;
import ru.yandex.partner.testapi.fixture.Fixture;
import ru.yandex.partner.testapi.fixture.FixtureContext;
import ru.yandex.partner.testapi.fixture.FixtureResult;
import ru.yandex.partner.testapi.fixture.service.idprovider.IdProviderService;
import ru.yandex.partner.testapi.fixture.service.tus.TusService;
import ru.yandex.partner.testapi.fixture.service.tus.TusUser;
import ru.yandex.partner.testapi.utils.ResourceUtils;

import static ru.yandex.partner.dbschema.partner.tables.UserAdfox.USER_ADFOX;
import static ru.yandex.partner.dbschema.partner.tables.UserFeatures.USER_FEATURES;
import static ru.yandex.partner.dbschema.partner.tables.UserRole.USER_ROLE;
import static ru.yandex.partner.dbschema.partner.tables.Users.USERS;
import static ru.yandex.partner.testapi.fixture.service.idprovider.IdClassEnum.ADFOX_ID;
import static ru.yandex.partner.testapi.fixture.service.idprovider.IdClassEnum.CLIENT_ID;

public abstract class AbstractUserFixture implements Fixture {
    private final LocalDateTime localDateTime = LocalDateTime.of(2020, 3, 31, 12, 0);
    private final LocalDateTime endDateTime = LocalDateTime.of(2023, 3, 31, 12, 0);

    protected final DSLContext dslContext;
    private final TusService tusService;
    private final IdProviderService idProviderService;
    private final ObjectMapper objectMapper;
    private final Boolean isTusUserNeeded;

    protected AbstractUserFixture(DSLContext dslContext,
                                  TusService tusService,
                                  IdProviderService idProviderService,
                                  ObjectMapper objectMapper,
                                  Boolean isTusUserNeeded) {
        this.dslContext = dslContext;
        this.tusService = tusService;
        this.idProviderService = idProviderService;
        this.objectMapper = objectMapper;
        this.isTusUserNeeded = isTusUserNeeded;
    }

    public abstract Logger getLogger();

    public UserOptsHelper getUserOpts() {
        UserOptsHelper userOpts = new UserOptsHelper();
        userOpts.setHasRsya(true);
        return userOpts;
    }

    private UserOptsHelper getUserOpts(JsonNode node) {
        var userOptsHelper = getUserOpts();
        if (node != null && node.get("addons") != null) {
            try {
                ObjectReader optReaderForUpdate = objectMapper.readerForUpdating(userOptsHelper);
                userOptsHelper = optReaderForUpdate.readValue(node.get("addons"));
            } catch (IOException e) {
                throw new TestApiException("Error during parse opts", e);
            }
        }
        return userOptsHelper;
    }


    public String serializeUserOpts(UserOptsHelper userOpts) {
        ObjectMapper mapper = new ObjectMapper();

        try {
            return mapper.writeValueAsString(userOpts);
        } catch (JsonProcessingException e) {
            getLogger().error("Error UserOpts serializing");
            throw new RuntimeException("Error UserOpts serializing");
        }
    }

    public String canonizeLogin(String login) {
        return login
                .replaceAll("\\.", "-")
                .toLowerCase();
    }

    protected UsersRecord getDefaultUserRecord() {
        UsersRecord usersRecord = new UsersRecord();
        usersRecord.setClientId(idProviderService.getNextId(CLIENT_ID));
        usersRecord.setName("name");
        usersRecord.setLastname("lastname");
        usersRecord.setMidname("midname");
        usersRecord.setEmail("mocked-email@ya.ru");
        usersRecord.setPhone("+799912312312");
        usersRecord.setAccountantEmail("mocked-accountant-email@ya.ru");
        usersRecord.setNewsletter(0L);
        usersRecord.setMultistate(0L);
        usersRecord.setCountryId(225L);
        usersRecord.setBusinessUnit(0L);
        usersRecord.setNoStatMonitoringEmails(1L);
        usersRecord.setIsTutby(0L);
        usersRecord.setCreateDate(localDateTime);
        usersRecord.setBlockLightFormEnabled(0L);
        usersRecord.setLastPayout(localDateTime);
        usersRecord.setNeedToEmailProcessing(0L);
        usersRecord.setIsMobileMediation(0L);
        usersRecord.setIsVideoBlogger(0L);
        usersRecord.setIsAdfoxPartner(0L);
        usersRecord.setIsGames(0L);
        return usersRecord;
    }

    // call getDefaultUserRecord and update result
    // Beware of dots in TusUser login. Replace it with "-".
    protected UsersRecord getUserRecord() {
        return getDefaultUserRecord();
    }

    public List<Long> getRoleIds() {
        return Collections.emptyList();
    }

    @Override
    public List<String> getFixtureDepends() {
        return Collections.emptyList();
    }

    @Override
    public List<FixtureResult> createAndSave(JsonNode optsJson, FixtureContext fixtureContext) {
        Opts opts = getOpts(optsJson);

        UsersRecord usersRecord = getUserRecord();
        Map<String, Object> fixtureResultParams = new HashMap<>();

        if (isTusUserNeeded) {
            TusUser tusUser = tusService.getOrCreateUser();
            String canonizedLogin = canonizeLogin(tusUser.getLogin());
            usersRecord.setId(tusUser.getUid());
            usersRecord.setUid(tusUser.getUid());
            usersRecord.setLogin(canonizedLogin);
            usersRecord.setDomainLogin("domain-" + canonizedLogin);

            fixtureResultParams.put("password", tusUser.getPassword());
        }

        usersRecord.setOpts(serializeUserOpts(getUserOpts(optsJson)));
        dslContext.insertInto(USERS)
                .set(usersRecord)
                .execute();
        //TODO
//        if (usersRecord.getId() == null) {
//            // для некоторых типов пользователей (напрмер INTAPI_USER_ID) id был установлен заранее
//            var id = dslContext
//                    .select()
//                    .from(USERS)
//                    .orderBy(USERS.ID.desc())
//                    .limit(1)
//                    .fetchOne(USERS.ID);
//            usersRecord.setId(id);
//        }

        fixtureResultParams.put("id", usersRecord.getId());
        fixtureResultParams.put("login", usersRecord.getLogin());

        if (usersRecord.getIsAdfoxPartner() != 0) {
            UserAdfoxRecord record = new UserAdfoxRecord();
            record.setUserId(usersRecord.getId());
            record.setCreateDate(localDateTime);
            record.setAdfoxId(idProviderService.getNextId(ADFOX_ID));
            record.setAdfoxLogin(usersRecord.getLogin());
            dslContext.insertInto(USER_ADFOX)
                    .set(record)
                    .execute();
        }

        insertUserRoles(usersRecord.getId());

        insertUserFeatures(opts, usersRecord.getId());

        updateAdditionalTables(usersRecord, fixtureContext);

        return List.of(
                new FixtureResult(
                        usersRecord.getId().toString(),
                        fixtureResultParams
                )
        );
    }

    protected void updateAdditionalTables(UsersRecord usersRecord, FixtureContext fixtureContext) {
        // overrride in implementations
    }

    protected DSLContext getDslContext() {
        return dslContext;
    }

    protected void insertUserRoles(Long userId) {
        List<UserRoleRecord> userRoleRecords = getRoleIds().stream()
                .map(roleId -> {
                    UserRoleRecord record = new UserRoleRecord();
                    record.setUserId(userId);
                    record.setRoleId(roleId);
                    return record;
                })
                .collect(Collectors.toList());

        JooqUtils.insertRecords(dslContext, USER_ROLE, userRoleRecords);
    }

    private void insertUserFeatures(Opts opts, Long userId) {
        if (opts != null) {
            List<UserFeaturesRecord> userFeaturesRecords = opts.getUserFeaturesRecords(userId);
            JooqUtils.insertRecords(dslContext, USER_FEATURES, userFeaturesRecords);
        }
    }

    protected void insertContract(UsersRecord usersRecord) {
        ContractsRecord contract = new ContractsRecord();
        contract.setId(0L);
        contract.setMultistate(0L);
        contract.setType("PARTNERS");
        contract.setExternalId("РС-21500-07/14");
        contract.setClientId(usersRecord.getClientId());
        contract.setDt(localDateTime);
        contract.setEndDt(endDateTime);
        contract.setUpdateDt(localDateTime);
        try {
            String contractJson = ResourceUtils.readAsString("/data/fixtures/contract/contract.json");
            Map<String, Object> contractMap = objectMapper.readValue(contractJson, new TypeReference<>() { });
            contractMap.put("client_id", usersRecord.getClientId());
            contractJson = objectMapper.writeValueAsString(contractMap);
            contract.setContract(contractJson);

            contractJson = ResourceUtils.readAsString("/data/fixtures/contract/contract_opts.json");
            contractMap = objectMapper.readValue(contractJson, new TypeReference<>() { });
            Map<String, Object> rawContract = (Map<String, Object>) contractMap.get("raw_contract");
            rawContract.put("client_id", usersRecord.getClientId());
            contractJson = objectMapper.writeValueAsString(contractMap);
            contract.setOpts(contractJson);
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Json processing error", e);
        }

        getDslContext().executeInsert(contract);
    }

    private Opts getOpts(JsonNode json) {
        try {
            return json != null
                    ? objectMapper.treeToValue(json, Opts.class)
                    : null;
        } catch (IOException e) {
            throw new TestApiException("Error during parse opts", e);
        }
    }

    protected String readOptsFromResource() {
        return ResourceUtils.readAsString("/data/fixtures/" + getFixtureName() + "/opts.json");
    }

    @SuppressWarnings("unused")
    private static class Opts {
        private List<String> features;

        List<String> getFeatures() {
            return features;
        }

        void setFeatures(List<String> features) {
            this.features = features;
        }

        List<UserFeaturesRecord> getUserFeaturesRecords(Long userId) {
            return features == null
                    ? Collections.emptyList()
                    : features.stream()
                    .map(feature -> {
                        UserFeaturesRecord record = new UserFeaturesRecord();
                        record.setUserId(userId);
                        record.setFeature(feature);
                        return record;
                    })
                    .collect(Collectors.toList());
        }
    }

}
