package ru.yandex.partner.testapi.fixture.widget;


import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.jooq.DSLContext;
import org.jooq.impl.SQLDataType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.partner.dbschema.partner.enums.StatisticsReportsReportType;
import ru.yandex.partner.dbschema.partner.tables.records.FormDataRecord;
import ru.yandex.partner.dbschema.partner.tables.records.StatisticsReportsRecord;
import ru.yandex.partner.dbschema.partner.tables.records.WidgetsRecord;
import ru.yandex.partner.testapi.fixture.FixtureContext;
import ru.yandex.partner.testapi.fixture.FixtureResult;
import ru.yandex.partner.testapi.utils.ResourceUtils;

import static ru.yandex.partner.dbschema.partner.tables.FormData.FORM_DATA;
import static ru.yandex.partner.dbschema.partner.tables.StatisticsReports.STATISTICS_REPORTS;
import static ru.yandex.partner.dbschema.partner.tables.Widgets.WIDGETS;

@Component
public class AllWidgetsFixture extends WidgetFixture {
    private static final String FIXTURE_NAME = "all_widgets";

    @Autowired
    public AllWidgetsFixture(DSLContext dslContext, ObjectMapper objectMapper) {
        super(dslContext, objectMapper);
    }

    @Override
    public List<String> getFixtureDepends() {
        return List.of(getUserFixtureName());
    }



    public Long getUserId(FixtureContext fixtureContext) {
        List<FixtureResult> fixtureResult = fixtureContext.valueByFixtureName(getUserFixtureName());
        if (fixtureResult == null || fixtureResult.isEmpty()) {
            throw new RuntimeException("All widgets fixture fail");
        }
        return (Long) fixtureResult.get(0).getParams().get("id");
    }

    public String getNextStatisticsReportsId() {
        StatisticsReportsRecord statisticsReportsRecord = dslContext
                .selectFrom(STATISTICS_REPORTS)
                .where("id regexp '^[0-9]+$'")
                .orderBy(STATISTICS_REPORTS.ID.cast(SQLDataType.BIGINT).desc())
                .limit(1)
                .fetchOne();
        if (statisticsReportsRecord != null) {
            String str = statisticsReportsRecord.getId();
            return String.valueOf(Integer.parseInt(str) + 1);
        } else {
            return "1";
        }
    }

    public Long getLastFormDataId() {
        return dslContext
                .selectFrom(FORM_DATA)
                .orderBy(FORM_DATA.ID.desc())
                .limit(1)
                .fetchOne()
                .getId();
    }


    @Override
    public String getFixtureName() {
        return FIXTURE_NAME;
    }

    @Override
    public List<FixtureResult> createAndSave(JsonNode optsJson, FixtureContext fixtureContext) {
        Long formDataId = updateFormData(fixtureContext);
        String statisticsReportId = updateStatisticReports(fixtureContext);
        Long userId = updateWidgets(fixtureContext, statisticsReportId);
        return List.of(
                new FixtureResult(
                        userId.toString(),
                        Map.of(
                                "widget", userId,
                                "statistics_report", statisticsReportId,
                                "form_data", formDataId
                        )
                )
        );
    }

    // Нужен для появления у пользователя виджета widget_welcome
    private Long updateFormData(FixtureContext fixtureContext) {
        FormDataRecord formDataRecord = new FormDataRecord();
        formDataRecord.setUserId(getUserId(fixtureContext));
        dslContext
                .insertInto(FORM_DATA)
                .set(formDataRecord)
                .execute();

        return getLastFormDataId();
    }

    private Long updateWidgets(FixtureContext fixtureContext, String statisticsReportId) {
        WidgetsRecord widgetsRecord = new WidgetsRecord();
        widgetsRecord.setUserId(getUserId(fixtureContext));
        String widgetsSettingsAsString = ResourceUtils.readAsString("/data/fixtures/all_widgets/widgets.json");
        try {
            List<Map<String, Object>> widgetList =
                    objectMapper.readValue(widgetsSettingsAsString, new TypeReference<>() { });
            HashMap<String, String> settings = (HashMap<String, String>) widgetList.get(0).get("settings");
            settings.put("report_id", statisticsReportId);
            settings = (HashMap<String, String>) widgetList.get(1).get("settings");
            settings.put("report_id", statisticsReportId);
            widgetsRecord.setJsonWidgetsSettings(objectMapper.writeValueAsString(widgetList));
        } catch (JsonProcessingException e) {
            e.printStackTrace();
        }
        widgetsRecord.setJsonDefaultDeletedIds(
                ResourceUtils.readAsString("/data/fixtures/all_widgets/deleted_widgets.json")
        );
        widgetsRecord.setMultistate(2L);
        dslContext
                .insertInto(WIDGETS)
                .set(widgetsRecord)
                .execute();

        return widgetsRecord.getUserId();
    }

    private String updateStatisticReports(FixtureContext fixtureContext) {
        StatisticsReportsRecord statisticsReportsRecord = new StatisticsReportsRecord();
        statisticsReportsRecord.setId(getNextStatisticsReportsId());
        statisticsReportsRecord.setCaptionRu("Тестовый отчет");
        statisticsReportsRecord.setCaptionEn("Test report");
        statisticsReportsRecord.setLevel("payment");
        statisticsReportsRecord.setQuery(
                ResourceUtils.readAsString("/data/fixtures/all_widgets/statistics_report_query.json")
        );
        statisticsReportsRecord.setOwnerId(getUserId(fixtureContext));
        statisticsReportsRecord.setNotShow(0L);
        statisticsReportsRecord.setMultistate(3L);
        statisticsReportsRecord.setReportType(StatisticsReportsReportType.mol);
        statisticsReportsRecord.setCategoryBits(7L);
        statisticsReportsRecord.setReportOrder(0L);
        dslContext
                .insertInto(STATISTICS_REPORTS)
                .set(statisticsReportsRecord)
                .execute();

        return statisticsReportsRecord.getId();
    }
}
