package ru.yandex.partner.testapi.service.testcase;

import java.io.IOException;
import java.io.InputStream;
import java.sql.SQLIntegrityConstraintViolationException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.jooq.DSLContext;
import org.jooq.exception.DataAccessException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.BeanInitializationException;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;

import ru.yandex.partner.libs.memcached.MemcachedService;
import ru.yandex.partner.testapi.exceptions.TestApiException;
import ru.yandex.partner.testapi.fixture.Fixture;
import ru.yandex.partner.testapi.fixture.FixtureResult;
import ru.yandex.partner.testapi.fixture.service.idprovider.IdProviderService;
import ru.yandex.partner.testapi.fixture.service.tus.TusService;
import ru.yandex.partner.testapi.jooq.LoggingExecuteListener;

import static ru.yandex.partner.dbschema.partner.tables.KvStore.KV_STORE;

/**
 * Все методы потоко НЕ БЕЗОПАСНЫ
 */

public class BaseTestApiService implements TestApiService {
    private static final Logger LOGGER = LoggerFactory.getLogger(TestApiService.class);
    private static final int DELAY_MINUTES_COUNT = 1;
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    private final DSLContext dsl;
    private final LoggingExecuteListener loggingExecuteListener;
    private final Map<String, Fixture> fixtureMap;
    private final TusService tusService;
    private final Set<String> dictionaryTables;
    private final IdProviderService idProviderService;
    private final ObjectMapper objectMapper;
    private final LocalDateTime initTime;
    private final String allTablesQuery;
    private final MemcachedService memcachedService;
    private LocalDateTime updateTime;
    private MultiValueMap<String, FixtureResult> fixtureStateMap;

    public BaseTestApiService(DSLContext dsl,
                              LoggingExecuteListener loggingExecuteListener,
                              List<Fixture> fixtures,
                              TusService tusService,
                              Set<String> dictionaryTables,
                              IdProviderService idProviderService,
                              MemcachedService memcachedService) {
        this.dsl = dsl;
        this.loggingExecuteListener = loggingExecuteListener;

        this.fixtureMap = fixtures.stream().collect(Collectors.toMap(
                Fixture::getFixtureName,
                Function.identity(),
                (fixture, fixture2) -> {
                    LOGGER.error("Fixtures name must be unique. Duplicated FixtureName = {}",
                            fixture.getFixtureName());
                    throw new BeanInitializationException("Fixtures name must be unique. Duplicated FixtureName = "
                            + fixture.getFixtureName());
                }));
        this.tusService = tusService;
        this.dictionaryTables = dictionaryTables;
        this.idProviderService = idProviderService;

        this.initTime = getInitTime(dsl);
        this.updateTime = initTime;

        this.fixtureStateMap = new LinkedMultiValueMap<>();
        this.objectMapper = new ObjectMapper();
        this.memcachedService = memcachedService;

        this.allTablesQuery = "SELECT table_name FROM information_schema.tables WHERE table_type='BASE TABLE' " +
                "AND table_schema='partner' AND table_name NOT IN (" +
                dictionaryTables.stream().map(t -> "?").collect(Collectors.joining(",")) + ");";
        LOGGER.info("Initialize fixtures: {}", this.fixtureMap.keySet());
        LOGGER.info("DB dictionary tables: {}", this.dictionaryTables);
    }

    /**
     * Получаем время запуска для получения списка таблиц, которые нужно очистить.
     * Ищем в таблице kv_store
     * Если записи нет - сохраняем текущую дату/время
     *
     * @return LocalDateTime с временем запуска
     */
    private LocalDateTime getInitTime(DSLContext dsl) {
        String dbInittime = dsl.select()
                .from(KV_STORE)
                .where(KV_STORE.KEY.eq("testapi_db_inittime"))
                .fetchOne(KV_STORE.VALUE, String.class);
        if (dbInittime == null) {
            throw new TestApiException("Use old autotest DB image without inittime value.");
        }
        LocalDateTime dbInitTimeValue = LocalDateTime.parse(dbInittime, DATE_TIME_FORMATTER);
        LOGGER.info("InitTime found in KV_STORE. Value {}", dbInitTimeValue);
        // Добавляем задержку после сборки базы
        return dbInitTimeValue.plusMinutes(DELAY_MINUTES_COUNT);
    }

    @Override
    public void processResourceTestcase(String testcase) {
        LOGGER.info("Process testcase: {}", testcase);

        String resource = "testcases/" + testcase + ".json";
        try (InputStream inputStream = ClassLoader.getSystemResourceAsStream(resource)) {
            if (inputStream == null) {
                LOGGER.error("Testcase '{}' resource not found ", testcase);
                throw new IllegalArgumentException("Testcase resource not found " + testcase);
            }
            FixtureListDto fixtureListDto = objectMapper.readValue(inputStream, FixtureListDto.class);
            processFixtureList(fixtureListDto);
        } catch (IOException e) {
            LOGGER.error("Error during I/O operation with testcase '{}'", testcase, e);
            throw new TestApiException("Error during I/O operation " + resource, e);
        }
    }

    private void processDatabaseBeforeFixture(boolean fullClean) {
        LOGGER.info("set sql_mode = '', set foreign_key_checks=0");
        dsl.execute("SET sql_mode=''");


        dsl.execute("set foreign_key_checks=0");
        prepareDatabaseBeforeTestcase(fullClean);

        // Убираем секунду так как в бд время без миллисикунд
        updateTime = LocalDateTime.now().minusSeconds(1L);
    }

    private void processDatabaseAfterFixture() {
        LOGGER.info("set foreign_key_checks=1");
        dsl.execute("set foreign_key_checks=1");
    }

    @Override
    public void processFixtureList(FixtureListDto fixtureListDto) {
        dsl.transaction(configuration -> {
            try {
                processFixtureListInternal(fixtureListDto, false);
            } catch (DataAccessException e) {
                if (e.getCause() instanceof SQLIntegrityConstraintViolationException) {
                    tryLog(e, updateTime);
                    processFixtureListInternal(fixtureListDto, true);
                } else {
                    throw e;
                }
            }
        });
    }

    public void processFixtureListInternal(FixtureListDto fixtureListDto, boolean fullClean) {
        LOGGER.info("Process fixture list {}", fixtureListDto);

        processDatabaseBeforeFixture(fullClean);
        memcachedService.flush();

        fixtureStateMap = new LinkedMultiValueMap<>();
        tusService.clearUserCachePosition();
        idProviderService.clearIdMap();

        FixtureContextImpl fixtureContext = new FixtureContextImpl();

        Map<String, FixtureDto> fixtureMetaMap = fixtureListDto.getFixtures()
                .stream()
                .collect(Collectors.toMap(FixtureDto::getName, Function.identity()));

        var container = new ArrayList<String>();
        loggingExecuteListener.start(container);
        try {
            for (FixtureDto fixture : fixtureListDto.getFixtures()) {
                processFixture(
                        fixture,
                        fixtureMetaMap,
                        new LinkedHashSet<>(),
                        fixtureContext
                );
            }
        } catch (DataAccessException e) {
            LOGGER.error("Tried insert : {}; Error : {}", container, e.getMessage());


            container.stream()
                    .distinct()
                    .forEach(tableName -> {
                                try {
                                    LOGGER.info(dsl.fetch("select * from " + tableName).toString());
                                } catch (Exception ex) {
                                    LOGGER.warn("Cannot select data from {}. ERROR: {}", tableName, ex.getMessage());
                                }
                            }
                    );
            throw e;
        } finally {
            loggingExecuteListener.stop();
        }

        processDatabaseAfterFixture();
    }

    @Override
    public MultiValueMap<String, FixtureResult> getFixtureStateMap() {
        return fixtureStateMap;
    }

    private void processFixture(FixtureDto fixtureDto,
                                Map<String, FixtureDto> fixtureDtoMap,
                                LinkedHashSet<String> dependsPath,
                                FixtureContextImpl fixtureContext) {

        LOGGER.info("Process fixture {}", fixtureDto);

        if (!LocalDateTime.now().isAfter(initTime)) {
            throw new TestApiException("Too early fixture run. Please wait one minute after autotest DB init.");
        }


        String fixtureName = fixtureDto.getName();
        if (fixtureContext.contains(fixtureName)) {
            LOGGER.info("Fixture {} already processed", fixtureName);
            return;
        }

        verifyDependsPath(fixtureName, dependsPath);
        dependsPath.add(fixtureName);

        Fixture fixture = getVerifiedFixture(fixtureName);


        List<String> fixtureDepends = fixture.getFixtureDepends();
        LOGGER.info("Process fixture depends: {}", fixtureDepends);

        for (String fixtureDependName : fixtureDepends) {
            processFixture(
                    fixtureDtoMap.getOrDefault(fixtureDependName, new FixtureDto(fixtureDependName)),
                    fixtureDtoMap,
                    dependsPath,
                    fixtureContext
            );
        }
        List<FixtureResult> result = fixture.createAndSave(fixtureDto.getOptsJson(), fixtureContext);
        LOGGER.info("Get fixture result: {}", result);

        fixtureContext.addValue(fixtureName, result);
        fixtureStateMap.addAll(fixtureName, result);

        dependsPath.remove(fixtureName);
    }

    private void prepareDatabaseBeforeTestcase(boolean fullClean) {

        Collection<String> tablesForTruncate;
        if (fullClean) {
            tablesForTruncate = getAllTables();
        } else {
            tablesForTruncate = getModifiedTables();
        }

        LOGGER.info("Tables for truncate: {}", tablesForTruncate);
        truncateTables(tablesForTruncate);
    }

    private Set<String> getModifiedTables() {
        String sql = "SELECT table_name FROM information_schema.tables" +
                " WHERE table_schema = 'partner' AND table_type='BASE TABLE' AND update_time > ?;";
        var tableNames = dsl.fetch(sql, updateTime).stream()
                .map(r -> r.get(0, String.class))
                .filter(Predicate.not(dictionaryTables::contains))
                .collect(Collectors.toSet());
        LOGGER.info("Selected tables by information_schema to truncate: {}", tableNames);
        return tableNames;
    }

    private Set<String> getAllTables() {
        // оставлю выборку тут, а не в конструкторе
        // на случай если таблицы создаются в процессе (или накатка миграций, которая проходит параллельно)
        var tableNames = dsl.fetch(allTablesQuery, dictionaryTables.toArray()).stream()
                .map(r -> r.get(0, String.class))
                .filter(Predicate.not(dictionaryTables::contains))
                .collect(Collectors.toSet());
        LOGGER.info("Selected ALL Tables to truncate: {}", tableNames);
        return tableNames;
    }

    private void tryLog(DataAccessException exception, LocalDateTime searchTime) {
        LOGGER.warn("The database is probably not completely cleaned up", exception);
        try {
            // Пытаемся вырезать название таблицы из сообщения вида:
            // SQL [insert into `user_features` (`user_id`, `feature`) values (?, ?)];
            // Duplicate entry '1490608884-context_on_site_natural' for key 'uniq_user_features__user_id_feature'
            var message = exception.getMessage();
            var startText = "insert into `";
            var startIndex = message.indexOf(startText) + startText.length();
            var errorTable = message.substring(startIndex, message.indexOf("`", startIndex));

            var record = dsl.fetchOne("SELECT update_time FROM information_schema.tables " +
                    "WHERE table_schema = 'partner' AND table_name = ?;", errorTable);

            if (record == null) {
                LOGGER.warn("Cannot get update_time for table = '{}'", errorTable);
                return;
            }

            LOGGER.warn("Search update_time is '{}'. {} update_time is '{}'",
                    searchTime, errorTable, record.get(0, String.class));


        } catch (Exception e) {
            LOGGER.warn("Failed to fetch table from error string: '{}'", exception.getMessage(), e);
        }
    }

    private void truncateTables(Collection<String> tableNames) {
        LocalDateTime startedAt = LocalDateTime.now();
        for (String tableName : tableNames) {
            dsl.truncate(tableName).execute();
        }
        LOGGER.info("Tables truncate took {}", Duration.between(startedAt, LocalDateTime.now()));
    }

    private void verifyDependsPath(String fixtureName, Set<String> dependsPath) {
        if (dependsPath.contains(fixtureName)) {
            LOGGER.error("Cycling fixture depends. Path = {}", Arrays.toString(dependsPath.toArray()));
            throw new TestApiException("Cycling fixture depends. Path = " + Arrays.toString(dependsPath.toArray()));
        }
    }

    private Fixture getVerifiedFixture(String fixtureName) {
        Fixture fixture = fixtureMap.get(fixtureName);
        if (fixture == null) {
            LOGGER.error("Testcase contains nonexistent fixture. FixtureName = {}", fixtureName);
            throw new TestApiException("Testcase contains nonexistent fixture. FixtureName = " + fixtureName);
        }
        return fixture;
    }
}
