package ru.yandex.partner.coreexperiment;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.InsertOnDuplicateSetStep;
import org.jooq.InsertValuesStepN;
import org.jooq.Record;
import org.jooq.SortField;
import org.jooq.TableField;
import org.jooq.impl.DSL;
import org.jooq.impl.TableImpl;
import org.jooq.util.mysql.MySQLDSL;

import ru.yandex.partner.coreexperiment.dbrequest.DBQuery;

public abstract class AbstractRepository<R extends Record, M extends ModelInterface> {
    private final DSLContext dsl;

    private final TableImpl<R> table;

    private final List<TableField<R, ?>> tableFields;

    private final ConverterInterface<R, M> recordToModelConverter;

    private final ConverterInterface<M, R> modelToRecordConverter;

    public AbstractRepository(
            DSLContext dsl,
            TableImpl<R> table,
            List<TableField<R, ?>> tableFields,
            ConverterInterface<R, M> recordToModelConverter,
            ConverterInterface<M, R> modelToRecordConverter
    ) {
        this.dsl = dsl;
        this.table = table;
        this.tableFields = tableFields;
        this.recordToModelConverter = recordToModelConverter;
        this.modelToRecordConverter = modelToRecordConverter;
    }

    protected abstract void setPk(M model, R record);

    protected abstract Condition getPkCondition(Object id);

    public M findById(Object id) {
        DBQuery query = new DBQuery().setCondition(this.getPkCondition(id));

        List<M> items = find(query);

        return items.get(0);
    }

    public List<M> find(DBQuery query) {

        return dsl.selectFrom(table)
                .where(query.hasCondition() ? query.getCondition() : null)
                .orderBy(query.hasOrderFields() ? query.getOrderFields() : new SortField[0])
                .limit(query.hasLimit() ? DSL.param("limit", query.getLimit()) : null)
                .offset(query.hasOffset() ? DSL.param("offset", query.getOffset()) : null)
                .fetch(recordToModelConverter::convert);
    }

    /**
     * <p>The method adds an item to a database and sets a primary key</p>
     * @param item
     */

    public void add(M item) {
        R record = modelToRecordConverter.convert(item);

        InsertValuesStepN<R> insert = getQueryInsert(List.of(record));

        R result = insert.returning(table.getPrimaryKey().getFields()).fetchOne();

        this.setPk(item, result);
    }

    public void store(List<M> items) {
        List<R> records = items.stream().map(modelToRecordConverter::convert)
                .collect(Collectors.toList());

        InsertValuesStepN<R> insert = getQueryInsert(records);

        InsertOnDuplicateSetStep<R> insertUpdate = insert.onDuplicateKeyUpdate();

        Map<TableField<R, ?>, Field<?>> fieldValue = tableFields.stream().collect(
                Collectors.toMap(f -> f, MySQLDSL::values));

        insertUpdate.set(fieldValue).execute();
    }

    private InsertValuesStepN<R> getQueryInsert(List<R> records) {
        InsertValuesStepN<R> insert = dsl.insertInto(table,
                tableFields);

        records.forEach(r -> insert.values(tableFields.stream().map(r::get).collect(Collectors.toList())));

        return insert;
    }
}
