package ru.yandex.partner.core.action;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.action.exception.ActionError;
import ru.yandex.partner.core.action.log.ActionsLogger;
import ru.yandex.partner.core.action.result.ActionsResult;
import ru.yandex.partner.core.entity.IncomingFields;
import ru.yandex.partner.core.exceptions.ValidationException;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.utils.converter.MassResultConverter;

public abstract class AbstractActionContextWithModelProperty<T extends ModelWithId>
        implements ActionContextWithModelProperty<T, ActionModelContainerImpl<T>> {
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractActionContextWithModelProperty.class);

    // Минимальный набор полей для всех контейнеров.
    // В некоторых контейнерах могут быть и другие поля
    private final Set<ModelProperty<?, ?>> requiredFields;
    private final Map<Long, ActionModelContainerImpl<T>> containers;
    private final ActionsLogger actionsLogger;
    private final ActionErrorHandler<T> errorHandler;

    protected AbstractActionContextWithModelProperty(
            ActionsLogger actionsLogger,
            ActionErrorHandler<T> errorHandler) {

        this.actionsLogger = actionsLogger;
        this.errorHandler = errorHandler;
        this.requiredFields = new HashSet<>();
        this.containers = new HashMap<>();
    }

    /**
     * Метод наливает по необходимости данные в контейнеры и возвращает их
     *
     * @param ids           - список id моделей
     * @param fields        - поля необходимые для наполнения в моделях
     * @param withoutErrors - возвращает все контейнеры или только те в сущностях которых не было ошибок
     * @return список контейнеров
     */
    @Override
    public List<ActionModelContainerImpl<T>> getContainers(
            Collection<Long> ids,
            Set<ModelProperty<? extends Model, ?>> fields,
            Boolean withoutErrors) {
        return getContainers(ids, fields, withoutErrors, CoreFilterNode.neutral());
    }

    /**
     * Наливает контейнеры для найденных по id и фильтру сущностей.
     * Внизу будет сделан фильтр [ID in ids], который смержится с параметром filter
     */
    @Override
    public List<ActionModelContainerImpl<T>> getContainers(
            Collection<Long> ids, Set<ModelProperty<?, ?>> fields, Boolean withoutErrors, CoreFilterNode<T> filter) {
        if (!TransactionSynchronizationManager.isActualTransactionActive()) {
            //мы не в транзакции, значит нужно читать постоянно
            //и ничего не хранить
            return StreamEx.of(createContainersByIds(ids, fields, filter).values()).toList();
        }
        return getOrFillContainers(ids, withoutErrors, fields, filter);
    }

    @Override
    public List<ActionModelContainerImpl<T>> getContainersWithoutErrors(Collection<Long> ids) {
        return getContainers(ids, requiredFields, true);
    }

    @Override
    public Set<Long> entityIdsWithErrors() {
        return StreamEx.of(errorHandler.getDefects().entrySet()).filter(it -> !it.getValue().isEmpty())
                .map(Map.Entry::getKey).toSet();
    }

    @Override
    public boolean hasErrors() {
        return errorHandler.hasErrors();
    }

    @Override
    public Map<Long, List<ActionError>> getErrors() {
        return errorHandler.getDefects();
    }

    @Override
    public List<ModelChanges<T>> getModelChangesToUpdate() {
        List<ActionModelContainerWithModelProperty<T>> actionModelContainers = containers.values().stream()
                .filter(container -> !errorHandler.entityHasError(container.getItem().getId()))
                .collect(Collectors.toList());

        List<ModelChanges<T>> modelChangesList = new ArrayList<>(actionModelContainers.size());

        for (ActionModelContainerWithModelProperty<T> actionModelContainer : actionModelContainers) {
            ModelChanges<T> modelChanges = this.createModelChanges(actionModelContainer.getItem().getId());

            var changedFields = actionModelContainer.getChangedFields();

            changedFields.forEach(field -> modelChanges.process(field.get(actionModelContainer.getItem()),
                    (ModelProperty<Model, Object>) field));

            if (modelChanges.isAnyPropChanged()) {
                modelChangesList.add(modelChanges);

                actionModelContainer.resetChangedFields();
            }
        }

        return modelChangesList;
    }

    @Override
    public void init() {
    }

    @Override
    public void commit(ActionsResult<?> result, IncomingFields incomingFields, boolean rollbackIfErrors) {
        List<ModelChanges<T>> modelChanges = this.getModelChangesToUpdate();

        try {
            if (!modelChanges.isEmpty()) {
                var operation =
                        this.getUpdateOperation(modelChanges, incomingFields, rollbackIfErrors);
                MassResult<T> curMassResult = MassResultConverter.convertMassResult(
                        operation.prepare().orElseGet(operation::apply));
                result.putToResults(getEntityClass(), curMassResult);
                if (!curMassResult.isSuccessful() || curMassResult.getErrorCount() > 0) {
                    var vr = (ValidationResult<List<T>, Defect>) curMassResult.getValidationResult();
                    errorHandler.addValidationErrors(vr);
                }
            }
        } catch (ValidationException validationException) {
            LOGGER.error("Validation exception caught during commit of entity {} context",
                    getEntityClass(), validationException
            );
            var affectedEntityIds = validationException.getAffectedIds()
                    .get(getEntityClass());

            if (affectedEntityIds != null) {
                affectedEntityIds.forEach(entityId ->
                        errorHandler.addErrorById(entityId, validationException.getDefectInfo(),
                                ActionError.ActionDefectType.OTHER)
                );
            } else {
                // fail all containers
                containers.values().forEach(container ->
                        errorHandler.addErrorById(container.getItem().getId(), validationException.getDefectInfo(),
                                ActionError.ActionDefectType.OTHER)
                );
            }
        }

        if (errorHandler.hasErrors() && rollbackIfErrors) {
            actionsLogger.clear();
        } else {
            this.commitContainers();
            actionsLogger.commit();
        }
    }

    @Override
    public void preloadModels(Set<ModelProperty<? extends Model, ?>> preloadedProperties,
                              List<T> preloadedModels) {
        for (T preloadedModel : preloadedModels) {
            containers.put(preloadedModel.getId(), new ActionModelContainerImpl<>(
                    preloadedModel,
                    clone(preloadedModel),
                    preloadedProperties
            ));
        }
    }

    //TODO: make it generic PI-24241
    protected abstract List<T> getModelsByIds(
            Collection<Long> ids, Set<ModelProperty<? extends Model, ?>> fields,
            CoreFilterNode<T> filter, boolean forUpdate
    );

    protected abstract T clone(T item);

    protected abstract ModelChanges<T> createModelChanges(Long id);

    private void commitContainers() {
        actionsLogger.drop(
                containers.values().stream()
                        .map(it -> it.getItem().getId())
                        .filter(errorHandler::entityHasError)
                        .iterator()
        );
    }

    @Override
    public void addFieldsToRequired(Collection<ModelProperty<?, ?>> fields) {
        requiredFields.addAll(fields);
    }

    private Map<Long, ActionModelContainerImpl<T>> createContainersByIds(
            Collection<Long> ids, Set<ModelProperty<? extends Model, ?>> fields, CoreFilterNode<T> filter) {

        // берем блокировку, т.к. модель еще не читалась из базы в текущей транзакции
        List<T> elements = getModelsByIds(ids, fields, filter, true);
        return StreamEx.of(elements).collect(Collectors.toMap(ModelWithId::getId,
                elem -> new ActionModelContainerImpl<>(
                        elem,
                        clone(elem),
                        fields
                )
        ));
    }

    public int getContextSize() {
        return containers.size();
    }

    private List<ActionModelContainerImpl<T>> getOrFillContainers(
            Collection<Long> ids, boolean withoutErrors, Set<ModelProperty<?, ?>> fields, CoreFilterNode<T> filter) {
        List<ActionModelContainerImpl<T>> result = new ArrayList<>(ids.size());

        List<Long> presentIds = ids.stream()
                .filter(containers::containsKey)
                .filter(id -> !(withoutErrors && errorHandler.entityHasError(id)))
                .collect(Collectors.toList());

        var allRequiredFields = new HashSet<>(requiredFields);
        allRequiredFields.addAll(fields);

        Set<ModelProperty<?, ?>> nonExistingFields = presentIds.stream()
                .map(containers::get)
                .flatMap(c -> c.getNonExistingFields(allRequiredFields).stream())
                .collect(Collectors.toSet());

        List<Long> idsWithNonExistingFields = presentIds.stream()
                .map(containers::get)
                .filter(c -> !c.getNonExistingFields(nonExistingFields).isEmpty())
                .map(c -> c.getItem().getId())
                .collect(Collectors.toList());

        if (!idsWithNonExistingFields.isEmpty()) {
            // блокировка не нужна, модель уже есть в контейнере. дозапрашиваем поля
            List<T> dbModels = getModelsByIds(
                    idsWithNonExistingFields, nonExistingFields, filter, false
            );

            dbModels.forEach(m ->
                    containers.get(m.getId()).setAllNonExistedProperties(nonExistingFields, m)
            );
        }

        presentIds.forEach(id ->
                result.add(containers.get(id))
        );

        List<Long> notFoundsIds = ids.stream()
                .filter(id -> !containers.containsKey(id))
                .collect(Collectors.toList());

        if (!notFoundsIds.isEmpty()) {
            Map<Long, ActionModelContainerImpl<T>> newContainers = createContainersByIds(
                    notFoundsIds, allRequiredFields, filter
            );
            containers.putAll(newContainers);
            result.addAll(StreamEx.of(newContainers.values()).toList());
        }

        return result;
    }

    @VisibleForTesting
    public Set<ModelProperty<? extends Model, ?>> getRequiredFields() {
        return requiredFields;
    }

    @Override
    public ActionsLogger actionLogger() {
        return actionsLogger;
    }
}
