package ru.yandex.partner.core.action;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.partner.core.action.exception.ActionError;
import ru.yandex.partner.core.action.exception.ActionRuntimeException;
import ru.yandex.partner.core.action.exception.presentation.ActionDefectMsg;
import ru.yandex.partner.core.action.log.ActionsLogger;
import ru.yandex.partner.core.action.result.ActionsResult;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.StateFlag;
import ru.yandex.partner.core.validation.defects.DefectInfoBuilder;
import ru.yandex.partner.libs.i18n.MsgWithArgs;
import ru.yandex.partner.libs.multistate.action.ActionNameHolder;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;

public abstract class AbstractTransitionActionWithModelProperty<T extends ModelWithId, S extends StateFlag, R>
        implements TransitionActionWithModelProperty<T, ActionModelContainerImpl<T>, R>, NestedActionExecutor {
    private final ActionConfiguration<T, ?> parentFactory;
    private final String name;
    private final List<Long> containerIds;
    private final MultistateGraph<? super T, S> multistateGraph;
    private final ActionPerformer actionPerformer;
    private final ActionErrorHandler<T> actionErrorHandler;

    @SuppressWarnings("ParameterNumber")
    public AbstractTransitionActionWithModelProperty(
            ActionConfiguration<T, ?> parentFactory,
            String name,
            Collection<Long> containerIds,
            MultistateGraph<? super T, S> multistateGraph,
            ActionPerformer actionPerformer,
            ActionErrorHandler<T> actionErrorHandler) {

        this.parentFactory = parentFactory;
        this.name = name;
        this.containerIds = new ArrayList<>(containerIds);
        this.multistateGraph = multistateGraph;
        this.actionPerformer = actionPerformer;
        this.actionErrorHandler = actionErrorHandler;
    }

    @Override
    public List<Long> getContainerIds() {
        return containerIds;
    }

    @Override
    public void check(List<ActionModelContainerImpl<T>> actionModelContainers) {
        checkByAction(actionModelContainers);
        var filteredContainers = StreamEx.of(actionModelContainers).filter(it ->
                !getActionErrorHandler().entityHasError(it.getItem().getId())).toList();
        List<Boolean> results =
                multistateGraph.checkActionAllowed(this.getName(),
                        filteredContainers.stream().map(ActionModelContainer::getItem)
                                .collect(Collectors.toList()));

        for (int i = 0; i < results.size(); i++) {
            if (!results.get(i)) {
                var item = filteredContainers.get(i).getItem();
                actionErrorHandler.addErrorById(item.getId(),
                        DefectInfoBuilder.createDefectInfo(MsgWithArgs.of(ActionDefectMsg.CAN_NOT_DO_ACTION,
                                getName())), ActionError.ActionDefectType.CAN_NOT_DO_ACTION);
            }
        }
    }

    /**
     * при необходимости можно установить дополнительную проверку на уровне конкрентного action
     * реализуйте этот метод и добавьте ошибку в errorhandler по конкретной сущности
     */
    protected void checkByAction(List<ActionModelContainerImpl<T>> containers) {
        // реализация в наследниках
    }

    @Override
    public void changeMultistate(List<ActionModelContainerImpl<T>> actionModelContainers) {
        Map<S, Boolean> flagModificationsValues = multistateGraph.getFlagModificationsForAction(name);
        actionModelContainers.forEach(container -> {
            Multistate<S> oldMultistate = getMultistate(container.getItem());

            Multistate<S> newMultistate = oldMultistate.copy();

            if (!(flagModificationsValues == null || flagModificationsValues.isEmpty())) {
                newMultistate.setFlags(flagModificationsValues);

                setMultistate(container, newMultistate);
            }
        });
    }


    public abstract Multistate<S> getMultistate(T model);

    public abstract void setMultistate(ActionModelContainerWithModelProperty<T> container, Multistate<S> multistate);

    @Override
    public void prepareContext(ActionContextWithModelProperty<T, ?> actionContext) {
        Set<ModelProperty<?, ?>> filteredFields = this.getDependsOn();
        actionContext.addFieldsToRequired(filteredFields);
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public ActionConfiguration<T, ?> getConfiguration() {
        return parentFactory;
    }

    @Override
    public Set<ModelProperty<?, ?>> getDependsOn() {
        return parentFactory.getDependsOnByClass(this.getEntityClass());
    }

    /**
     * метод вызывается при необходимости в методе onActionHook, использует те же контейнеры, контексты и обработчики
     * ошибок, что и родительский метод, потому что выполняются в том же потоке
     *
     * @return возвращает текущий результат выполнения всех action этого потока
     */
    @SafeVarargs
    @Override
    public final <M extends ModelWithId> ActionsResult<?> doNestedActions(boolean rollbackIfErrors, TransitionAction<M,
            ActionModelContainerImpl<M>, ?>... actions) {
        var allowedNestedActions = parentFactory.getAllowedNestedActions();

        var unknownActions = Arrays.stream(actions)
                .filter(action -> !allowedNestedActions.contains(action.getConfiguration()))
                .collect(Collectors.toList());

        if (!unknownActions.isEmpty()) {
            throw new ActionRuntimeException("Nested actions wasn't found. Action names = " +
                    Arrays.toString(unknownActions.toArray()));
        }

        return actionPerformer.doActions(rollbackIfErrors, actions);
    }

    /**
     * Признак необходимости записи в action-log.
     * На некоторых экшенах запись отключается, т.к. они слишком частые.
     */
    public boolean writeActionLog() {
        return true;
    }

    @Override
    public void makeLogging(ActionsLogger actionsLogger, ActionModelContainerImpl<T> container) {
        if (!writeActionLog()) {
            return;
        }
        Long modelId = container.getItem().getId();
        long oldMultistate = getMultistate(container.getNonChangedItem()).toMultistateValue();
        long newMultistate = getMultistate(container.getItem()).toMultistateValue();

        String opts = this.getSerializedOpts(modelId);
        actionsLogger.log(modelId, oldMultistate, this.getName(), newMultistate, opts);
    }

    protected List<Long> filteredItemIdsByActionAllowed(ActionNameHolder actionNameHolder,
                                                        List<ActionModelContainerImpl<T>> containers) {
        return getMultistateGraph()
                .filteredByActionAllowed(
                        actionNameHolder.getActionName(),
                        containers.stream()
                                .map(ActionModelContainerImpl::getItem)
                                .collect(Collectors.toList()))
                .stream()
                .map(t -> ((T) t).getId())
                .collect(Collectors.toList());
    }

    protected List<T> filteredItemsByActionAllowed(ActionNameHolder actionNameHolder,
                                                   List<T> items) {
        return getMultistateGraph()
                .filteredByActionAllowed(actionNameHolder.getActionName(), items)
                .stream().map(t -> (T) t)
                .collect(Collectors.toList());
    }

    protected MultistateGraph<? super T, S> getMultistateGraph() {
        return multistateGraph;
    }

    protected ActionErrorHandler<T> getActionErrorHandler() {
        return actionErrorHandler;
    }
}
