package ru.yandex.partner.core.action;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.ModelWithId;
import ru.yandex.partner.core.action.exception.ActionError;
import ru.yandex.partner.core.action.exception.ActionRuntimeException;
import ru.yandex.partner.core.action.result.ActionsResult;
import ru.yandex.partner.core.configuration.annotation.TransactionWithFallbackScope;
import ru.yandex.partner.core.entity.IncomingFields;

/**
 * Класс собирает все возможные контексты в Map<Class, Context>, выполняет actions по уровням вложенности и порядку,
 * Используется custom scope "transaction" - бин создается свой для каждой транзакции.
 */
@Component
@TransactionWithFallbackScope
public class ActionContextFacade {
    private static final Logger LOGGER = LoggerFactory.getLogger(ActionContextFacade.class);

    private final LinkedHashSet<Class<? extends ModelWithId>> contextsInOrder;
    private final Map<Class<? extends ModelWithId>,
            ActionContext<? extends ModelWithId,
                    ? extends ActionModelContainer<? extends ModelWithId>
                    >> actionContexts;
    private final IncomingFields updatedFields;
    private Long doActionDepth = 0L;

    @Autowired
    public ActionContextFacade(List<? extends ActionContext<
            ? extends ModelWithId,
            ? extends ActionModelContainer<? extends ModelWithId>>> actionContextList) {
        this.contextsInOrder = new LinkedHashSet<>();
        this.actionContexts = StreamEx.of(actionContextList).collect(Collectors.toMap(
                ActionContext::getEntityClass,
                Function.identity()
        ));
        this.updatedFields = new IncomingFields();
    }

    @VisibleForTesting
    public <T extends ModelWithId, A extends ActionContext<T, ?>>
    A getActionContext(Class<T> clazz, Class<A> knownActionClass) {
        return knownActionClass.cast(getActionContext(clazz));
    }

    public <T extends ModelWithId> ActionContextWithModelProperty<T, ?> getActionContext(Class<T> clazz) {
        var actionContext = Optional.ofNullable(actionContexts.get(clazz))
                .orElseThrow(() ->
                        new ActionRuntimeException("Action context %s wasn't found".formatted(clazz.getSimpleName()))
                );

        if (doActionDepth > 0) {
            // отслеживаем обращения только после вызова init()
            contextsInOrder.add(clazz);
        }

        return (ActionContextWithModelProperty<T, ?>) actionContext;
    }

    public void init() {
        doActionDepth++;
    }

    public void commit(ActionsResult<?> result, boolean rollbackIfErrors) {
        if (doActionDepth == 1) {
            // todo выявлять зацикливания или запретить повторный коммит одинаковых контекстов
            while (!contextsInOrder.isEmpty()) {
                var orderIterator = contextsInOrder.iterator();
                Class<? extends ModelWithId> ctxClass = orderIterator.next();
                orderIterator.remove();
                var context = this.actionContexts.get(ctxClass);
                context.commit(result, updatedFields, rollbackIfErrors);
                if (rollbackIfErrors && !result.getErrors().isEmpty()) {
                    break;
                }
            }
        }
        doActionDepth--;
    }

    public boolean hasErrors() {
        return StreamEx.of(actionContexts.values()).anyMatch(ActionContext::hasErrors);
    }

    public Map<Class<? extends ModelWithId>, Map<Long, List<ActionError>>> getAllErrors() {
        return StreamEx.of(actionContexts.values()).filter(ActionContext::hasErrors).collect(Collectors.toMap(
                ActionContext::getEntityClass, ActionContext::getErrors
        ));
    }

    @VisibleForTesting
    public Long geThreadInsideDoAction() {
        return doActionDepth;
    }

    public void addUpdatedFields(IncomingFields otherUpdatedFields) {
        this.updatedFields.merge(otherUpdatedFields);
    }
}
