package ru.yandex.partner.core.action;

import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;

public class ActionModelContainerImpl<T extends ModelWithId>
        implements ActionModelContainerWithModelProperty<T> {

    private final T item;
    private final T nonChangedItem;
    private final Set<ModelProperty<? super T, ?>> fields = new HashSet<>();
    private Set<ModelProperty<? super T, ?>> changedFields = new HashSet<>();

    public ActionModelContainerImpl(T item, T nonChangedItem, Set<ModelProperty<? extends Model, ?>> fields) {
        this.item = item;
        this.nonChangedItem = nonChangedItem;
        fields.forEach(it -> this.fields.add((ModelProperty<? super T, ?>) it));
    }

    @Override
    public <V> void changeProperty(ModelProperty<? super T, V> modelProperty, V value) {
        changedFields.add(modelProperty);
        fields.add(modelProperty);

        modelProperty.set(item, value);
    }

    protected <V> void setProperty(ModelProperty<? super T, V> modelProperty, V value) {
        fields.add(modelProperty);
        if (!changedFields.contains(modelProperty)) {
            modelProperty.set(item, value);
        }
        modelProperty.set(nonChangedItem, value);
    }

    @Override
    public T getItem() {
        return item;
    }

    @Override
    public T getNonChangedItem() {
        return nonChangedItem;
    }

    @Override
    public Set<ModelProperty<? super T, ?>> getNonExistingFields(
            Set<ModelProperty<? extends Model, ?>> checkFields) {
        return checkFields.stream().filter(field -> !fields.contains(field))
                .map(it -> (ModelProperty<? super T, ?>) it)
                .collect(Collectors.toSet());
    }

    public void setAllNonExistedProperties(Set<ModelProperty<? extends Model, ?>> checkFields,
                                           T model) {
        var nonExistingFields =
                this.getNonExistingFields(checkFields);
        for (var field : nonExistingFields) {
            this.setProperty((ModelProperty<? super T, Object>) field,
                    ((ModelProperty<? super T, Object>) field).get(model));
        }
    }

    @Override
    public Set<ModelProperty<? super T, ?>> getChangedFields() {
        return changedFields;
    }

    @Override
    public <V> V getProperty(ModelProperty<? super T, V> modelProperty) {
        return modelProperty.get(item);
    }

    @Override
    public void resetChangedFields() {
        changedFields = new HashSet<>();
    }

}
