package ru.yandex.partner.core.action;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import com.google.common.collect.Maps;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.partner.core.action.result.ActionsResult;
import ru.yandex.partner.libs.annotation.PartnerTransactional;

@Service
public class ActionPerformerImpl implements ActionPerformer {
    private final ActionContextFacade actionContextFacade;

    @Autowired
    public ActionPerformerImpl(ActionContextFacade actionContextFacade) {
        this.actionContextFacade = actionContextFacade;
    }

    @PartnerTransactional
    @Override
    public <T extends ModelWithId, C extends ActionModelContainer<T>, R>
    ActionsResult<R> doActions(TransitionAction<T, C, R> action) {
        return doActions(true, action);
    }

    @PartnerTransactional
    @Override
    public <T extends ModelWithId, C extends ActionModelContainer<T>, R>
    ActionsResult<R> doActions(boolean rollbackIfErrors, TransitionAction<T, C, R> action) {
        var listActionsResult = doActionsInternal(rollbackIfErrors, List.of(action));
        return ActionsResult.asSingleResult(listActionsResult);
    }

    @PartnerTransactional
    @Override
    public <T extends ModelWithId, C extends ActionModelContainer<T>>
    ActionsResult<?> doActions(TransitionAction<T, C, ?>... actions) {
        return doActionsInternal(true, List.of(actions));
    }

    @PartnerTransactional
    @Override
    public <T extends ModelWithId, C extends ActionModelContainer<T>>
    ActionsResult<?> doActions(
            boolean rollbackIfErrors, TransitionAction<T, C, ?>... actions) {
        return doActionsInternal(rollbackIfErrors, List.of(actions));
    }

    @PartnerTransactional
    @Override
    public <T extends ModelWithId, C extends ActionModelContainer<T>> ActionsResult<?> doActions(
            boolean rollbackIfErrors, List<TransitionAction<T, C, ?>> actions) {
        return doActionsInternal(rollbackIfErrors, actions);
    }

    private <T extends ModelWithId, C extends ActionModelContainer<T>> ActionsResult<List<Object>> doActionsInternal(
            boolean rollbackIfErrors, List<TransitionAction<T, C, ?>> actions) {
        var result = new ActionsResult<List<Object>>();
        actionContextFacade.init();
        StreamEx.of(actions)
                .select(WithUpdatedFields.class)
                .map(WithUpdatedFields::getIncomingFields)
                .forEach(actionContextFacade::addUpdatedFields);

        result.setResponseData(executeActions(result, actions));

        actionContextFacade.commit(result, rollbackIfErrors);
        result.setCommitted(true);
        if (actionContextFacade.hasErrors()) {
            result.addAllErrors(actionContextFacade.getAllErrors());
            if (rollbackIfErrors) {
                result.setCommitted(false);
                TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            }
        }
        return result;
    }

    private <T extends ModelWithId, C extends ActionModelContainer<T>> List<Object> executeActions(
            ActionsResult<?> actionsResult,
            List<TransitionAction<T, C, ?>> actions) {
        Map<Class<T>, Set<Long>> ids = Maps.newHashMapWithExpectedSize(actions.size());
        for (TransitionAction<T, C, ?> action : actions) {
            ActionContextWithModelProperty<T, ?> actionContext = actionContextFacade
                    .getActionContext(action.getEntityClass());

            // взаимопомощь
            // сначала action кладет необходимые данные в контекст
            // после контекст обогащает их
            action.prepareContext(actionContext);

            ids.computeIfAbsent(action.getEntityClass(),
                            k -> new TreeSet<>())
                    .addAll(action.getContainerIds());
        }

        ids.forEach((modelClass, id) ->
                actionContextFacade.getActionContext(modelClass).getContainersWithoutErrors(id));

        var actionResults = new ArrayList<>(actions.size());
        for (TransitionAction<T, C, ?> action : actions) {
            ActionContext<T, C> actionContext = (ActionContext<T, C>)
                    actionContextFacade.getActionContext(action.getEntityClass());

            var containers = actionContext.getContainersWithoutErrors(action.getContainerIds());

            action.check(containers);
            var idsWithErrors = actionContext.entityIdsWithErrors();
            var validContainers = StreamEx.of(containers).filter(
                    it -> !idsWithErrors.contains(it.getItem().getId())).toList();

            action.changeMultistate(validContainers);
            validContainers.forEach(c -> action.makeLogging(actionContext.actionLogger(), c));

            Object actionResult = action.onActionWithResult(actionContext, validContainers);

            // assuming MassResult<T>
            if (actionResult instanceof MassResult) {
                actionsResult.putToResults(action.getEntityClass(), (MassResult<? extends ModelWithId>) actionResult);
            }

            actionResults.add(actionResult);
        }

        return actionResults;
    }

    @Override
    public ActionContextFacade getActionContextFacade() {
        return actionContextFacade;
    }

}
