package ru.yandex.partner.core.action;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.jooq.DSLContext;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.repository.TypeModifyRepository;
import ru.yandex.direct.multitype.repository.container.RepositoryContainer;
import ru.yandex.direct.multitype.service.type.update.UpdateOperationContainer;
import ru.yandex.direct.multitype.service.type.update.UpdateOperationTypeSupportFacade;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.Operation;
import ru.yandex.partner.core.action.log.ActionsLogger;
import ru.yandex.partner.core.entity.IncomingFields;
import ru.yandex.partner.core.entity.ModelQueryService;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.multitype.service.validation.type.ValidationTypeSupportFacade;
import ru.yandex.partner.core.operation.PartnerTypedUpdateOperation;

public abstract class TypeSupportActionContext<B extends ModelWithId, T extends B,
        C extends UpdateOperationContainer<B> & RepositoryContainer>
        extends AbstractActionContextWithModelProperty<T> {
    private final Class<T> entityClass;
    // TODO extract interface with locking query method,
    //  both ModelQueryService and TypedRepository comply
    private final ModelQueryService<? super T> queryService;
    private final Function<T, T> copyFunction;
    private final DSLContext dslContext;
    private final UpdateOperationTypeSupportFacade<B, C, C, C> updateOperationTypeSupport;
    private final TypeModifyRepository<B, B, C, C> modifyRepository;
    private final ValidationTypeSupportFacade<B, C, C> validationTypeSupportFacade;
    private final Supplier<C> editContainerCtor;

    @SuppressWarnings("ParameterNumber")
    protected TypeSupportActionContext(
            Class<T> entityClass,
            ActionsLogger actionsLogger,
            ActionErrorHandler<T> actionErrorHandler,
            ModelQueryService<? super T> queryService,
            Function<T, T> copyFunction,
            DSLContext dslContext,
            UpdateOperationTypeSupportFacade<B, C, C, C> updateOperationTypeSupport,
            TypeModifyRepository<B, B, C, C> modifyRepository,
            ValidationTypeSupportFacade<B, C, C> validationTypeSupportFacade,
            Supplier<C> editContainerCtor
    ) {
        super(actionsLogger, actionErrorHandler);
        this.entityClass = entityClass;
        this.queryService = queryService;
        this.copyFunction = copyFunction;
        this.dslContext = dslContext;
        this.updateOperationTypeSupport = updateOperationTypeSupport;
        this.modifyRepository = modifyRepository;
        this.validationTypeSupportFacade = validationTypeSupportFacade;
        this.editContainerCtor = editContainerCtor;
    }

    @Override
    protected List<T> getModelsByIds(Collection<Long> ids,
                                             Set<ModelProperty<? extends Model, ?>> fields,
                                             CoreFilterNode<T> filter,
                                             boolean forUpdate) {
        return queryService.findAll(
                QueryOpts.forClass(getEntityClass())
                        .withFilterByIds(ids)
                        .withFilter(filter)
                        .withProps(fields)
                        .forUpdate(forUpdate)
        );
    }

    @Override
    protected T clone(T item) {
        return copyFunction.apply(item);
    }

    @Override
    protected ModelChanges<T> createModelChanges(Long id) {
        return new ModelChanges<>(id, getEntityClass());
    }

    @Override
    public Class<T> getEntityClass() {
        return entityClass;
    }

    @Override
    public Operation<Long> getUpdateOperation(List<ModelChanges<T>> modelChanges, IncomingFields updatedFields,
                                              boolean rollbackIfErrors) {
        return new PartnerTypedUpdateOperation<>(
                getEntityClass(),
                rollbackIfErrors ? Applicability.FULL : Applicability.PARTIAL,
                modelChanges.stream()
                        .map(userModelChanges -> userModelChanges.castModel(getEntityClass()))
                        .collect(Collectors.toList()),
                Set.of(),
                dslContext,
                (Collection<Long> ids) -> getContainersWithoutErrors(new ArrayList<>(ids))
                        .stream()
                        .map(ActionModelContainerImpl::getNonChangedItem)
                        .collect(Collectors.toList()),
                updateOperationTypeSupport,
                modifyRepository,
                validationTypeSupportFacade,
                editContainerCtor.get()
        );
    }

}
