package ru.yandex.partner.core.action.factories;

import java.lang.reflect.Field;
import java.util.ArrayList;

import javax.annotation.PostConstruct;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.model.ModelWithId;
import ru.yandex.partner.core.action.ActionConfiguration;
import ru.yandex.partner.core.action.ActionPayloadType;
import ru.yandex.partner.core.action.TransitionAction;

/**
 * Маркерный интерфейс для фабрик action-ов
 */
public abstract class ActionFactory<M extends ModelWithId, A extends TransitionAction<M, ?, ?>>
        implements ActionConfigurationFactory<M, A> {
    private static final Logger LOGGER = LoggerFactory.getLogger(ActionFactory.class);
    private final ActionConfigurationFactory<M, A> actionConfigurationFactory;

    ActionFactory(ActionConfigurationFactory<M, A> actionConfigurationFactory) {
        // only package-private create
        // нужно тк на типы наследников завязан код
        this.actionConfigurationFactory = actionConfigurationFactory;
    }

    @PostConstruct
    public void init() {
        if (!(actionConfigurationFactory instanceof SmartActionConfigurationLifecycle)) {
            return;
        }

        var smartActionConfigurationLifecycle = (SmartActionConfigurationLifecycle<M, A>) actionConfigurationFactory;

        var inaccessibleFields = new ArrayList<Field>();

        for (Field field : getClass().getDeclaredFields()) {
            Object fieldValue = null;
            try {
                field.setAccessible(true);
                fieldValue = field.get(this);
            } catch (IllegalAccessException e) {
                inaccessibleFields.add(field);
            }

            if (fieldValue instanceof ActionConfigurationFactory) {
                smartActionConfigurationLifecycle.getBuilder()
                        .allowNestedInternal(((ActionConfigurationFactory<?, ?>) fieldValue).getActionConfiguration());
            }
        }

        if (!inaccessibleFields.isEmpty()) {
            LOGGER.error("Could not access bean \"{}\" fields {}", getClass(), inaccessibleFields);
        }

        smartActionConfigurationLifecycle.complete();
    }

    @Override
    public ActionConfiguration<M, A> getActionConfiguration() {
        return actionConfigurationFactory.getActionConfiguration();
    }

    public abstract ActionPayloadType getType();

    /**
     * @return Название action-а
     */
    public abstract String getName();

    public abstract Class<M> getModelClass();

    /**
     * Определяет список полей, который необходимо отдать клиенту при исполнении action.
     * Актуально только для {@link ActionPayloadType#CUSTOM}.
     */
    public FieldSetType getReturnFields() {
        return FieldSetType.DEFAULT;
    }
}
