package ru.yandex.partner.core.action.log;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.InsertSetMoreStep;
import org.jooq.Record;
import org.jooq.Table;

import ru.yandex.partner.core.action.ActionUserIdContext;

@ParametersAreNonnullByDefault
public abstract class AbstractActionsLogger<R extends Record> implements ActionsLogger {
    private final DSLContext dslContext;
    private final Table<R> table;
    private Map<Long, List<R>> records = new HashMap<>();
    private final ActionUserIdContext actionUserIdContext;

    public AbstractActionsLogger(DSLContext dslContext, Table<R> table,
                                 ActionUserIdContext actionUserIdContext) {
        this.dslContext = dslContext;
        this.table = table;
        this.actionUserIdContext = actionUserIdContext;
    }

    protected abstract R getRecord(Long userId, Long modelId, long oldMultistate, String actionName,
                                   long newMultistate, String opts);

    @Override
    public void commit() {
        var iterator = records.values().stream().flatMap(Collection::stream).iterator();
        if (iterator.hasNext()) {
            InsertSetMoreStep<R> insertSetMoreStep = dslContext.insertInto(table).set(iterator.next());
            while (iterator.hasNext()) {
                insertSetMoreStep = insertSetMoreStep.newRecord().set(iterator.next());
            }
            insertSetMoreStep.execute();
            clear();
        }
    }

    @Override
    public void log(Long modelId, long oldMultistate,
                    String actionName, long newMultistate, String opts) {
        R record = getRecord(actionUserIdContext.get(), modelId, oldMultistate, actionName, newMultistate, opts);

        records.computeIfAbsent(modelId, k -> new ArrayList<>())
                .add(record);
    }

    @Override
    public void drop(Iterator<Long> ids) {
        ids.forEachRemaining(records::remove);
    }

    @Override
    public void clear() {
        this.records = new HashMap<>();
    }

    @Override
    public boolean hasRecords() {
        return !records.isEmpty();
    }
}
