package ru.yandex.partner.core.action.result;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.partner.core.action.exception.ActionError;

@ParametersAreNonnullByDefault
public class ActionsResult<X> {
    /**
     * Результат конкретного(ых) экшена(ов)
     */
    private X responseData;

    private boolean committed;

    /**
     * Ошибки, возникшие в процессе исполнения экшенов и валидации моделей
     */
    private Map<Class<? extends ModelWithId>, Map<Long, List<ActionError>>> errors;

    /**
     * Результаты дампов в БД в разрезе классов моделей
     */
    private Multimap<Class<? extends ModelWithId>, MassResult<? extends ModelWithId>> results;

    public ActionsResult() {
        this(
                new HashMap<>(),
                Multimaps.newListMultimap(new HashMap<>(), ArrayList::new)
        );
    }

    protected ActionsResult(
            Map<Class<? extends ModelWithId>, Map<Long, List<ActionError>>> errors,
            Multimap<Class<? extends ModelWithId>, MassResult<? extends ModelWithId>> results) {
        this.errors = errors;
        this.results = results;
    }

    public static <R> ActionsResult<R> asSingleResult(ActionsResult<List<Object>> listActionsResult) {
        Preconditions.checkState(listActionsResult.responseData.size() == 1,
                "List action result should have exactly one result");
        var singleResult = new ActionsResult<R>(listActionsResult.errors, listActionsResult.results);
        singleResult.responseData = (R) listActionsResult.responseData.get(0);
        singleResult.committed = listActionsResult.committed;
        return singleResult;
    }

    public boolean isCommitted() {
        return committed;
    }

    public void setCommitted(boolean committed) {
        this.committed = committed;
    }

    public Map<Class<? extends ModelWithId>, Map<Long, List<ActionError>>> getErrors() {
        return errors;
    }

    public void addAllErrors(Map<Class<? extends ModelWithId>, Map<Long, List<ActionError>>> errors) {
        StreamEx.of(errors.entrySet()).forEach(pair ->
                this.errors.put(pair.getKey(), new HashMap<>(pair.getValue())));
    }

    public void putToResults(Class<? extends ModelWithId> entityClass, MassResult<? extends ModelWithId> result) {
        results.put(entityClass, result);
    }

    public Multimap<Class<? extends ModelWithId>, MassResult<? extends ModelWithId>> getResults() {
        return results;
    }

    public <T extends ModelWithId> Collection<MassResult<T>> getResult(Class<T> entityClass) {
        return results.get(entityClass).stream()
                .map(it -> (MassResult<T>) it)
                .collect(Collectors.toList());
    }

    public void setResponseData(X actionsResult) {
        this.responseData = actionsResult;
    }

    public X getResponseData() {
        return responseData;
    }

    @Override
    public String toString() {
        return "ActionsResult{" +
                "responseData=" + responseData +
                ", committed=" + committed +
                ", errors=" + errors +
                ", results=" + results +
                '}';
    }
}
