package ru.yandex.partner.core.bs.enrich

import NPartner.Page.TPartnerPage
import NPartner.Page.TPartnerPage.TBlock
import com.google.common.annotations.VisibleForTesting
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Value
import org.springframework.context.MessageSource
import org.springframework.context.support.MessageSourceAccessor
import org.springframework.stereotype.Service
import ru.yandex.direct.model.Model
import ru.yandex.direct.model.ModelProperty
import ru.yandex.partner.core.CoreConstants
import ru.yandex.partner.core.block.BlockType
import ru.yandex.partner.core.bs.BkDataException
import ru.yandex.partner.core.bs.BkDataRepository
import ru.yandex.partner.core.bs.BkEnrichConfig
import ru.yandex.partner.core.bs.transport.ProtobufLogbrokerDto
import ru.yandex.partner.core.entity.QueryOpts
import ru.yandex.partner.core.entity.block.filter.BlockFilters
import ru.yandex.partner.core.entity.block.model.BaseBlock
import ru.yandex.partner.core.entity.block.service.BlockService
import ru.yandex.partner.core.entity.block.service.OperationMode
import ru.yandex.partner.core.entity.block.type.dsps.BlockWithDspsRepositoryTypeSupport.DSP_PROPS_FOR_BLOCK
import ru.yandex.partner.core.entity.dsp.model.Dsp
import ru.yandex.partner.core.entity.dsp.service.DspService
import ru.yandex.partner.core.entity.page.type.bkdata.PageBkDataValidatorProvider
import ru.yandex.partner.core.filter.CoreFilterNode
import ru.yandex.partner.core.multistate.block.BlockStateFlag
import ru.yandex.partner.libs.bs.json.BkDataConverter
import ru.yandex.partner.unifiedagent.UnifiedAgentResult
import ru.yandex.partner.unifiedagent.UnifiedAgentService
import java.io.IOException
import java.util.Optional

@Service
class BkDataService @Autowired constructor(
    private val blockService: BlockService,
    private val bkDataRepository: BkDataRepository<BaseBlock, TBlock>,
    messageSource: MessageSource,
    private val unifiedAgentService: UnifiedAgentService,
    @Value("\${bs.topic}") private val bsTopic: String?,
    private val pageBkDataValidatorProvider: PageBkDataValidatorProvider,
    private val dspService: DspService,
    private val enrichConfig: BkEnrichConfig
) {
    private val bkDataConverter = BkDataConverter()
    private val messages = MessageSourceAccessor(messageSource)

    @Throws(BkDataException::class)
    fun enrichPage(pageBkData: String?, readOnly: Boolean = true): TPartnerPage {
        val parseResult = try {
            bkDataConverter.convertPageJsonToProto(pageBkData)
        } catch (e: Exception) {
            throw BkDataException("Can't convert page bk data for page ${parsePageId(pageBkData)}", e)
        }
        if (parseResult.errors.isNotEmpty()) {
            throw BkDataException("Errors appeared during bkdata deserialization: " + parseResult.errors)
        }

        val tPartnerPage = parseResult.message
        val pageId = Optional.of(tPartnerPage.pageID)
            .filter { tPartnerPage.hasPageID() }
            .orElseThrow { BkDataException("Page has no id") }

        val blockIds = getSupportedBlocksForPages(listOf(pageId), setOf(BaseBlock.ID))
            .map { it.id }

        val blockBkDatas = bkDataRepository.getBkData(blockIds, readOnly)
        val resultPage = TPartnerPage.newBuilder(tPartnerPage)
        enrichPageWithBlocks(resultPage, blockBkDatas)
        return resultPage.build()
    }

    public fun convertToJson(tPartnerPage: TPartnerPage): String {
        return try {
            bkDataConverter.convertProtoToJson(tPartnerPage)
        } catch (e: IOException) {
            throw BkDataException(
                "Can't convert page ${tPartnerPage.pageID} to json after blocks enrichment", e
            )
        }
    }

    public fun send(page: TPartnerPage) {
        val dsps = dspService.findAll(QueryOpts.forClass(Dsp::class.java).withProps(DSP_PROPS_FOR_BLOCK))
            .associateBy { it.id }

        val validationResult = pageBkDataValidatorProvider.validator(OperationMode.CRON, dsps).apply(page)
        if (validationResult.hasAnyErrors()) {
            throw BkDataException(
                "PageBkDataValidator validation failed for page ${page.pageID}: ${validationResult.flattenErrors()}"
            )
        }

        val result = unifiedAgentService.sendToLogbroker(
            ProtobufLogbrokerDto(listOf(page.toByteString()), bsTopic)
        )
        if (result.status.equals(UnifiedAgentResult.Status.VALIDATION_ERROR)) {
            throw BkDataException("UnifiedAgent validation failed for page ${page.pageID}: ${result.details}")
        }
    }

    public fun getSupportedBlocksForPages(
        pageIds: Collection<Long>,
        props: Set<ModelProperty<out Model, *>>
    ): Collection<BaseBlock> {
        return enrichConfig.supportedBlockTypes.flatMap { blockType ->
            blockService.findAll(QueryOpts.forClass(
                blockType.constructor!!.get().javaClass
            )
                .withFilter(CoreFilterNode.and(
                    BlockFilters.PAGE_ID.`in`(pageIds),
                    BlockFilters.MULTISTATE.ne(BlockStateFlag.DELETED),
                    BlockFilters.MODEL.notIn(listOf(BlockType.CONTEXT_ON_SITE_CONTENT.literal)),
                ))
                .withProps(props)
            )
        }
    }

    public fun enrichPageWithBlocks(
        resultPage: TPartnerPage.Builder,
        blockBkDatas: List<TBlock>?
    ) {
        if (blockBkDatas == null) {
            return
        }
        enrichBlockData(resultPage, blockBkDatas)
        enrichRtbVideoData(resultPage, blockBkDatas)
    }

    private fun enrichBlockData(
        pageBuilder: TPartnerPage.Builder,
        blockBkDatas: List<TBlock>
    ) {
        val nonSupportedBlocks = pageBuilder.rtbBlocksList.stream()
            .filter { block: TBlock -> !isEnrichSupportedModel(block.blockModel) }
            .toList()
        pageBuilder.clearRtbBlocks()
            .addAllRtbBlocks(nonSupportedBlocks)
            .addAllRtbBlocks(blockBkDatas)
    }

    private fun enrichRtbVideoData(
        pageBuilder: TPartnerPage.Builder,
        blockBkDatas: List<TBlock>
    ) {
        if (pageBuilder.hasRtbVideo()) {
            return
        }
        val hasVideoType = blockBkDatas.stream()
            .filter { block: TBlock -> block.blockModel in enrichConfig.rtbVideoSupportedBlockTypesLiteral }
            .map { it.dspType }
            .anyMatch { type: Int -> (type and CoreConstants.DspTypes.DSP_VIDEO.bkId()) != 0 }
        if (hasVideoType) {
            val rtbVideoBuilder = pageBuilder.rtbVideoBuilder
            rtbVideoBuilder
                .setBufferEmptyLimit(5).bufferFullTimeout = 5000
            rtbVideoBuilder.addCategoriesBuilder()
                .setCategoryID(0)
                .setArchive(false).name = messages.getMessage(RtbVideoMsg.DOMAIN_OF_VIDEO_RESOURCE)
            rtbVideoBuilder
                .setPlatform(CoreConstants.VideoPlatforms.VIDEO_PLATFORM_HTML5.type)
                .setSkin(CoreConstants.DEFAULT_VIDEO_SKIN)
                .setSkinTimeout(5000)
                .setSkipDelay(5)
                .setSkipTimeLeftShow(true)
                .setTimeLeftShow(true)
                .setTitle(messages.getMessage(RtbVideoMsg.VIDEO_RESOURCES))
                .setVASTTimeout(5000)
                .setVPAIDEnabled(true)
                .setVPAIDTimeout(5000)
                .setVideoTimeout(5000)
                .setWrapperMaxCount(3).wrapperTimeout = 5000
        }
    }

    private fun isEnrichSupportedModel(model: String) = model in enrichConfig.supportedBlockTypesLiteral

    companion object {
        @VisibleForTesting
        fun parsePageId(rawJsonString: String?): Long? {
            val regex = "\"PageID\":\\s?\"(\\d+)\"".toRegex()
            return rawJsonString?.let { regex.find(rawJsonString)?.groupValues?.get(1) }?.toLong()
        }
    }
}
