package ru.yandex.partner.core.entity;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.meta.MetaFilter;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.core.utils.OrderBy;

import static ru.yandex.partner.core.filter.CoreFilterNode.and;

/**
 * Класс описывает набор параметров для запроса моделей
 * в сервисах.
 * @param <M>
 */
public class QueryOpts<M extends ModelWithId> implements QueryWithBatch<M>, DefaultQuery<M>, QueryWithIdsFilter<M> {
    private CoreFilterNode<? super M> filter;
    private Set<ModelProperty<?, ?>> props;
    private LimitOffset limitOffset;
    private List<OrderBy> orderByList;
    private Boolean forUpdate;
    private Integer batchSize;
    private Long lastId;
    private MetaFilter<? super M, Long> idMetaFilter;
    private Collection<Long> ids;
    private final Class<M> clazz;

    private QueryOpts(Class<M> clazz) {
        this.clazz = clazz;
    }

    /**
     * Entry point при создании объекта
     * @param clazz конечный класс для которого запрашиваются модели
     * @param <M>
     * @return
     */
    public static <M extends ModelWithId> QueryOpts<M> forClass(@NotNull Class<M> clazz) {
        return new QueryOpts<>(clazz);
    }

    @Override
    public QueryOpts<M> withFilter(@NotNull CoreFilterNode<? super M> filter) {
        this.filter = this.filter == null ? filter : and(this.filter, filter);
        return this;
    }

    @Override
    public QueryOpts<M> withProps(@Nullable Set<ModelProperty<? extends Model, ?>> props) {
        if (this.props == null || props == null) {
            this.props = props;
        } else {
            Set<ModelProperty<?, ?>> combinedProps = new HashSet<>();
            combinedProps.addAll(this.props);
            combinedProps.addAll(props);
            this.props = combinedProps;
        }
        return this;
    }

    @Override
    public DefaultQuery<M> withLimitOffset(@Nullable LimitOffset limitOffset) {
        this.limitOffset = limitOffset;
        return this;
    }

    @Override
    public DefaultQuery<M> withLimit(int limit) {
        return withLimitOffset(LimitOffset.limited(limit)); // argument validation inside
    }

    @Override
    public DefaultQuery<M> withOrder(@Nullable List<OrderBy> orderByList) {
        this.orderByList = orderByList;
        return this;
    }

    @Override
    public DefaultQuery<M> withOrder(@NotNull OrderBy... order) {
        return withOrder(List.of(order));
    }

    public QueryOpts<M> forUpdate(boolean forUpdate) {
        this.forUpdate = forUpdate;
        return this;
    }

    @Override
    public QueryOpts<M> forUpdate() {
        return forUpdate(true);
    }

    /**
     * @param previousBatch Последняя пачка моделей из findAll
     * @return флаг - получилось сдвинуть lastId или нет
     */
    @Override
    public boolean nextBatch(@NotNull List<M> previousBatch) {
        if (!isBatchOpts()) {
            throw new IllegalStateException("Create opts with batch size first");
        }
        if (!previousBatch.isEmpty() && previousBatch.get(previousBatch.size() - 1).getId() > lastId) {
            lastId = previousBatch.get(previousBatch.size() - 1).getId();
            return true;
        }
        return false;
    }

    @Override
    public QueryWithBatch<M> withBatch(int batchSize, @NotNull MetaFilter<? super M, Long> idMetaFilter) {
        if (batchSize < 0) {
            throw new IllegalArgumentException("batchSize must be non negative");
        }
        if (!idMetaFilter.getName().equals("id")) {
            throw new IllegalArgumentException("MetaFilter must be id, got " + idMetaFilter);
        }
        this.batchSize = batchSize;
        this.idMetaFilter = idMetaFilter;
        lastId = 0L;
        return this;
    }

    @NotNull
    @Override
    public CoreFilterNode<? super M> getFilter() {
        CoreFilterNode<? super M> result = filter != null ? filter : CoreFilterNode.neutral();
        if (isBatchOpts()) {
            result = and(result, CoreFilterNode.create(idMetaFilter, FilterOperator.GREATER, lastId));
        }
        return result;
    }

    @Nullable
    @Override
    public Set<ModelProperty<? extends Model, ?>> getProps() {
        return props;
    }

    @Nullable
    public LimitOffset getLimitOffset() {
        if (isBatchOpts()) {
            return new LimitOffset(batchSize, 0); // unpack it in repository method, offset would be ignored
        }
        return limitOffset;
    }

    @Nullable
    public List<OrderBy> getOrderByList() {
        return orderByList;
    }

    public boolean isForUpdate() {
        if (forUpdate == null) {
            return false;
        }
        return forUpdate;
    }

    @NotNull
    @Override
    public Class<M> getClazz() {
        return clazz;
    }

    public boolean isBatchOpts() {
        return batchSize != null && lastId != null && idMetaFilter != null;
    }

    @Override
    public QueryWithIdsFilter<M> withFilterByIds(Collection<Long> ids) {
        this.ids = ids;
        return this;
    }

    public boolean hasIds() {
        return ids != null;
    }

    public Collection<Long> getIds() {
        return ids;
    }
}
